//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////



`define STATE_SPI_WIDTH                 3
`define BITS_PER_WORD                   8
`define CPOL                            0
`define CPHA                            1
`define READ                            2
`define PAUSE                           3
`define WRITE                           4

module FPGA_spi #(
    parameter NUM_TARGETS = 1,
    localparam MAX_TARGET  = NUM_TARGETS -1
)
(
    // System
    input       clk                 ,
    input       rst_n               ,

    input [3:0] prescaler           ,//Indicates number of clocks for each edge of the SCLK, 0 = 1 clock cycle.  This allows slowing down the bus
    input [1:0] spi_mode            ,// {CPHA,CPOL}: CPOL has to change at least one clock cycle before a new command to allow correct SCK edge direction
    input [2:0] targ_sel            ,//Selects which downstream target to access.

    input       start               ,//command the start of a SPI transaction.  Must have write =1 at the same time
    input       write               ,//command a single byte write
    input       turnaround          ,//assert to trigger a turnaround in which clock is driven for <turnaround_len> cyle
    input       read                ,//command a single byte read
    input       stop                ,//indicate that the current OP is the end of the transaction


    input [1:0] spi_width           ,// 0x0 = Single SPI , 0x1 = Dual SPI , 0x2 = Quad SPI
    input [7:0] turnaround_len      ,//number of idle cycle for a turnaround 
    input [7:0] wr_data             ,//Data to be written.  This value is latched when write =1 and cmd_ack=1
    
    output      cmd_ack             , //pulse to indicate current command has been accepted and queued in the engine

    output[7:0] rd_data             , //last byte read on the spi interface
    output      rd_data_valid       , //strobe indicating a new valid value on rd_data for a read byte

    output      busy                , //0 = bus idle , 1 = transaction in process 

    
    output [MAX_TARGET:0] CS_N      , //SPI interface pins
    output SCK                      ,
    output       SDIO_en            , // Tri-State output enable.  "Pad" is at the top level.
    input  [3:0] SDIO_in            ,
    output [3:0] SDIO_out                

);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
jYmt1Mf/bd9HHrGDCW/KJrXNxO96x2qiBdnl9BtYEfPVe+Bc+Ic8WwioaNyd/OiW
GW+GzKFsO9RmLpCCTjE6GjLR7Jii7IjyXon3/Vt0lyP9GdLy/NpT0KkaJs9ap502
/HzqcuYPmhDmsHr2mFYQRRhcgTBnFgdB6u1dGP4srefCj4vk95MZktWBaH6XhbmI
U3ayyijxNGerympj8B1Y+c1tIvHwuG4aggxzscmcmgB+IuXUtd99Ev3tK6h9jhk2
ULDZrV7L/hsgVj+km0mC//cL+SneD6vrgnsUSXtponpYG6XRyjrMWJFRb6vtOeig
e6oCwxgpqXgYxqUDSn8tHg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
zCx6+JPdjCHG0PSYyxCQ0MVRdKvVIMqWLjufoDNu6M8wKNaWknEkrAYmnwZs/mpF
FFhtKbN4pDCrFlCuTOXgRUlpFkQOTIcRaaryFb/9Qk9nSJ+18M5Onsw7GH/1FoHS
9G/uPLGCTr4Vva2aiM7X32oUX3xDHSQCVD9w9tJU1qU=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
Vfs33ToArXI1UBfklC6xg/u8Kxmyzvsd64VjKIAXkMEsimpHiLIuxzHEJiIqz0v2
qPj/EdKCPAEv6zcfZ8hXhRZKtGmJEwQK0NcfkgC7UR9it1n2D9FFhZ70ONExlV05
XqbRaeqJpzkvhGsVQB39PyzNIGWjZqMO8NQ3AUfEqDhXGleThrAYJ+tvfzBUYEPz
wKYneNKsYYl/8BawB8mkriXeAoAsXUi7TOTzpfLRY8KMTrXdcNsQujkFemJ4Cv6H
n0MtNaly9CxwnLzqwLNMKj0PTQPn1pBvPiNaISVL4o0AMNRZYADSkEWW0DXaz/kc
Gvx8HjUkEPzMGr+ziRxSaw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
XTF6SBVtw9a2wl9D6nPGR4bkk6AGteOYo62rhdut47OJj0GyDLe88Vxc2De80T9e
c5DW3kht+AzwB0y4/EGT/23OzJQhXUnIU0gkhMX12PMFkHLQtxia+ehuYM89mdY5
FBtb5iEqkGT5dHaSrlxg3kEk0o3jbW37OtfAdzXrRVkb0UnC0AugYPfmv/ggUS23
u4hKWfv83dasMuCw4U18WXwK0IgtRCSIshPEnSIlsUCRCFHr3Lr8uq2jTOdrVQm3
1/3qLhuPSDdacOw8F5CAS0O0FrCTEug+S0p06OXclj/G8ENSc4fINufLT9a2/IYP
RUEXHCFx+BI/PHZs1Lkb8w==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
CiF2tHL9tHMhaDDRzdoqv514JxxB+wWK2FQQ7QxKtB8c+Vze1HPN4lkigEalaDLg
CTq9MCW0VOp/eg6H4dXopguwdRwz1Rpd1/hdxx0TpKwsliElbklxQXQAlCGtrPGU
iIblbNZsDO32cR/oyjisn2XxBd015QEfkJj3MnaXYlP0HUqdDq5LgLGtfNqZXuiP
iROF8dFfft5Bb511W9jsSf0mouDPIoOJdZ/hUUgKhKz2ZSXY5y3VVFqgiGiYEibD
wWAu8IYFRciUPJyAvyYub97J14G5IuUWWnxYKA8zMg9gcEUIToDPSFgsUO+DKwK+
A7EFhJAfB8ZBq6H9LcsZpQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
AN8p2e6viF7HVHFiHIT+BCEVcv7G40wWhI1sbTfScKGSRv7uRxc2PVwtKa+MSgs7
mTvxB7HVOeAYyPC5AhbJfarh22ZtPk7VLJB3PEZpQhwqhTWrwBiKR9RGoneRjobg
UhENwvaHPYyge9ELeHgC8l6OSGub7NyfbprsOtFv1gF5tbp8uCjU4AXcEK7AliGM
lrnPuQEfw1uOhXjIB9h0EtxtXdT3iQxvsLyuTnGoy/4cSVNsgbSTKcm7xZ+10RL1
OTwuySpi5ZXj9ZG0U72YBnbSbZ/jVrXdsn5aYFKxbBg/TnTmtWU4N0vd6Fypt+n9
bf7j9ITLvEpNABXS4D9bhA==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
LFpTtIALF0kJYvxLWGRxMaXoS0O7xjtF5l6HrVJ9ORSJzc1QkcOUmYzy/elWszor
hU6RUuM9qA9usCay9XF6/lA2Y1cyzc8H5S/bJz74Ce0IAo9RjeYmy1eYNLtaOggN
+eQ0obdXNxCt9AgQ1eFCnSD4aoW60WCPqGj6/R3bLmCAIfJWsilsjW2oXTU5iJcP
6E6UplVCcHXUXKqQMG6Lz/ftVrVGehPS1du0c+wW2HYdZgTSlxucLrJrWX5n4VTp
N9V8AiBJTRgrcxDYJf+ZRlgAeETQMavR2CAiPY/Zq3ir62LiLElVI6UKqx3L5qGp
bcvXDoc3AQWOvjI6fg3zeCX1PEQuNLOfE83r+vYypeHm9636tmId9B7nwrjGtUsH
b+g136wHuIpVlmleQWGMkwLVyHKG9obTDXV656SVuRFaT5crcnmZTxGNxIYJc29b
sbfdpDuzNOUqbAuMUA6ZfxwK6zIccgXXOQ/Wf4GoLT4f6AR7DtcWV1UkgwiRrPyY
Gu06HB9t6qK77yN9rCsQ5iq/o8E+4hfeAzxudBanFupXH4PWotuWdf8yjIFwA/ou
SKdadzsaV8zSsbqcHtNz1oOgImiacgmSPTWLMUTNjcOUBvJbXqUu7wD0oSenC5Gb
o9HaoP1PpGWs4LiPIH42skzCt142eWaKZp2c+DrKx/hdsi7d7ZpdI3wYaGxBDs/g
zV9F6LtS4BYN4ztMIzZg60Lj30H4Tc5uq518rdDb+2FbTdWGZG5Do4/Lmn91UHnE
vqfTKqH5GC5HzqyEgogCrzTevmm/vWeOVLl9dMpTVBtETlfOcQEP+e0sIkIdM6Sx
vcfkFmvqMUtsGWv0RXscg3eGNEgtJ+EZNq0PrsUiNEaDSEIxsf6uh/lZ0AMFK4HM
XOIC7IXZG+xx1TmG5GEO0I7Wm2LX4+1jymbIfy9/KWT5JUBKyV+ch5v37/MN2TZ7
+Xbjn5SzVfpnrEaFN2Dch3YzVZv8uHogoJnKde6I3AtwpMrui10IRZZohihmOfkK
w+ZFEBQztGLhIdR4FQVb8zcSpML8gVXl8cyqPyPxQw9zQxVBFwQ4/KcT3V7vZNTp
M/T0g3FtZc0SwQMrkzkt+Zonhz3JS3NXTGRHHJo0fqC8V7IuWzzjk6h1LSrMPz9y
lj9HWcuoLwKlCxsAophzaqc/DC6J3akVmxkZ7VpSjBj017Jf8ay9BDASP0OGfYoh
wn/uLpdXv/wEZAXRtJjOh3i3tTFIKvJIfyyUjTsyBUpKiyQBL5lkCKBp1hnp8DwB

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
az7Sd4PrgQ5R0UwkrY8fl1iA29ckzGCywwI5qwLUFiPo8HUD3VnPXqj+fv6fb89l
4I1VHQgHCvlXQjPENq39SktxzWXJMlE+XB/vKNeDfE4AANaJlfjU2Ky4DpB88wAm
a5XNx47hCMYYLlFkwqEqHArpaiwng31hIgK95L1cDbw=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=8800)
`pragma protect data_block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`pragma protect end_protected

endmodule
