//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////



module tx_stream_buffer
  import apb_pkg::*;
  import regmap_pkg::*;
  import axis_pkg::*;
#(
  parameter   HOST_DWIDTH       = 512,
  parameter   NUM_HOSTS         = 1,
  parameter   SENSOR_DWIDTH     = 128,
  parameter   FIFO_DEPTH        = 8192,
  localparam  HOST_KEEP_WIDTH   = (HOST_DWIDTH / 8),
  localparam  SENSOR_KEEP_WIDTH = (SENSOR_DWIDTH / 8),
  parameter   MTU               = 1500
) (
  //Clock and Reset
  input   logic                         i_hif_clk,
  input   logic                         i_hif_rst,
  input   logic                         i_sif_clk,
  input   logic                         i_sif_rst,
  input   logic                         i_apb_clk,
  input   logic                         i_apb_rst,
  //APB 
  input   apb_m2s                       i_apb_m2s,
  output  apb_s2m                       o_apb_s2m,
  //TX Sensor Data AXIS Input 
  input   logic                         i_axis_tvalid,
  input   logic [HOST_DWIDTH-1:0]       i_axis_tdata,
  input   logic                         i_axis_tlast,
  input   logic                         i_axis_tuser,
  input   logic [HOST_KEEP_WIDTH-1:0]   i_axis_tkeep,
  //TX Sensor Data AXIS Output
  output  logic                         o_axis_tvalid,
  output  logic [SENSOR_DWIDTH-1:0]     o_axis_tdata,
  output  logic                         o_axis_tlast,
  output  logic                         o_axis_tuser,
  output  logic [SENSOR_KEEP_WIDTH-1:0] o_axis_tkeep,
  input   logic                         i_axis_tready,
  //Pause
  output  logic [NUM_HOSTS-1:0]         o_eth_pause,
  output  logic [15:0]                  o_pause_quanta
);  





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
DmovO0ECF029ZI8nEqrnxsYAy8vM1oKCjfZTIX6F8w2zIvy35T+xjQeP2VOj59xk
OfnGct19/VXksPsozevcurCVP7LKVOv0/Z5jNe+KEZxg5MI1hH7Ady5LO9Ysy9cZ
wBqj3keAnq30ldaRj0f1H5TICxrXmUq6BnHApPQcupYm8Cooy1b9x/5G0YkrTVPX
qhCbGlXVt43fkZk7Nu+/vAjPdNHgWJJZqNWx50oielTCdWZoHxdjAnUc73aY6c/r
F5+jWInDTfdlfWA6qkd0P+s0oywifmpFad9iGo8Htk0iuNTt6JshcxpVcn7M6NIe
OFNzTNzthBvqVzgK+AoaFg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
nKS1qFelo2aC+gqhdu+/8raQqp2EFNpC7wO/U2wgmUqsox2aQZ98u9fCVoXNWDBA
zx7ZFUN2NkzfD0B6wa0kfUfPM/Ypr2VKlg3menFWokZaPs0JvldDXRsgC76VpSrY
jP7jgM5a/5FgYaMmHnhoZ4+bmsLpF32DDCxuSXZTzLo=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
eZEWamXahNHarrNMGZzt4ljCWvbc7sdB6HAg3rLB9lsYjDDphQQIXMQigPGLmq1q
QEsMIhb12wNEMtvm/3fJl99i+AtzSccElxpZ95r7u+faOha1m90dvZLxIlOROLaH
Vm+jMekwIuPxwfTF69WBdu3CVasBSdzDj5zTNsW2rOnhHEXMjqM92yR4SidNKSfe
ZY0VZjr3tfWBcJS859EjiUo2wdpn066O6UMbpoDXpNKRulvuQnpCAE0R08TWv3WP
wLnIhB96sgNKkQRBM95Gl5fbOJetT3EWkMtL8ulOagR2W1LO7KR4cGg0XYUuNRUE
E9jg+1VGTrFxrOSdz9m0tA==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
ms00pHXWfO6fxPEcnBlJYkV6seE0rYUgoITe85v9/nfGOEWTyAbIDbre5WloXEKe
o77CkI9RJLTZFcqizlrdsz1X8F2gnoWgl8kH1paayW9lqemXIEGi8sIm85ip/M9E
b2Aw++yoXyIRORSZ0p87A5jtg1HwJvhqXiFeOJL/hAILSocHPk1tzFayAv9M2n7x
eY0JcYcoOXdnvBnWLugOEoxiPFLMekn0qigfDuXIWXRvj2Z65DMp/PQ3O+zsWjn+
oQG08jgYyXqoOobTJgx+0XozHZfqxEA0tRcGs65Vu2R5NZmWXqftZ+b2hvnUL/S5
XVrT1kFFpm2YFsNDfsKZQg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
SAFdoMSYCHLm1svACxVhDm/C5cHKrxZ81uPOSS1DNko1i76YsJG0pFBwZ2nvh+jK
UAIexnP8guUnw8OM4MWZ8uTls6mkT5oHvHbg146Atj2OA9vPZNYlM8ViZi3UcMn4
/vydrPcKjZXEcGRWnHVcNSrYQODeHR2IdkWlYYOeK+RLPYbjwtOCWFvv4ViYxssS
JnNBDDYHGAUR5fWv8eGu7T0y7CHxZuLUB/9nqL+eqTETb+Z9GLVTqIevkE1ybzk2
P73uexMWiipDtdIpDuuAHJSJhPn767iXJSizDPFKBbom4xjMWuo0lvWZD4mrpsTj
C0r+HDGIMbgrQCPxlqnSCQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
JEKBOHmRnnTnxfaHFrqKGXIyhONPc7+GlNaGkUtqifN5lggr5G0tVJxWv08EWAGl
rBl1uDhyxVrqlE80Dvt1ehp6RBOmOH1D6/IxDKgx/i5OgMNU6AH8yHIaVU0zx+5N
PVDUfOV3UjO7YvWxEAORZDYiWSkfvEwJWFf2/m/TCK3j6+/v0tjw4M00kLeK3bJz
3LeNb2k2Pudpwi7gCQSbrr1cqHoffr4keXjtr23sPYWsC/ldFqiBzAAhw8B3JfB2
80wm8zYG872R90vvZ7y4B/Rfib3Ox0TpqNVVTlConGpUv/xd/T9+jj2sS57KjFh6
Idfryy/xuAs8VQK4wV77bw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
kjLAyTcbdgT2O0KxbuonfYikyCcDYPYk+4UYZrDTOAiFDwTb4vKppNq3c2feMySJ
V9Gk34mM6wX7PZX/QYXPqqEiIw8t+9+Mj6Q0goKjH7/dDuG6griWuug9BzkzS3s0
MV/9lSnWMd1IibqpJkAR7tjg7SMkjle2JtR5Pe26DVcgXiHr0YSATVwUgyZG8Dnt
UNtyZ3JZlSyMmKUKgjgfvdrYtpHQEOpok6UBTZAQrbIqLSTnL6mrpYj4nU7Yu2zs
qCYF358JRHRdKt3pIe6J4MzZqr3Ox0fD6uTLfEATPgBYuHNzOnicA2XBqN/rTpv3
WDIFNIYnzeckjFeza5I1RDwQd56Xdnpc11nToPOzg/LPeqmcx+6JYz2Xo68wRU4e
gCcifvY8X7KjoX2Omez2YdtV5fnwzsvjGSYaSUt3ASZXxG1UcHnEW6WQ7F3xtU7B
PID9Q0h0WftPjeY8HXxmGnAPOSC1R7vSsXgx2sceIproRSmCc9nku6wxJIwHjUhJ
lIqcivFukn6dSec/kRQM1MZFHwLy1f70TuL7kqkL917o1Rz/Rf7AAyvqkpZzYgXE
UNi1gJX9+PSb0NnWweEfgLqQMKf0qpOjZ4C2hRmSoXxaqHSAwi7tAncTVmuAqBJr
s3Wmf4ojyJ9GBxBuB8JuPtlaZv/bPs3rIi0mjp8TvdDHC8RznBuo/XSXogXicpYG
WnbvjsQo20lvjI7csxWk/RiFd04uQSv8BJLH1/QwehmQxsOqq+OcdbeAyn6LMdAz
7Ahn0yU2vuTt1OiufYpBXFGcEYh87vq6yvY2FlqrpHqc9/hlK9TpgFzcYBvvgNEN
cl1rOkihj9o6PUAYqPpN34w6MdXjffItCdYrCV3qGqe8oHJunzJyb7QCBX+TGMvc
cHaN/zUEigmfEmGmA6vl1VUBkaWOqXAmO5SL9vwHxmiEOWiSLRxfzn1rXtAZxabN
vYcqTNS8gt7SM/NDoQNoBRXcf3Gh+3R8+Zz8qNqAfAsoE+/0DTZR+pcalKuD6bf+
tNhOO+T9o7mjYRqI+/aqKkAvDtTLF1LZYVZZNwyb7Uk9egx1SQEiwGmKv6uK2vtK
rfp9+GiMfh9hdHlVBrD8o4UXXb7ZINqXACVLJgtv0sBg4tDY6oFa0G6y5WIb0zlb
HerC4symIYfFlqkciHvLPWbnnbYcaRi3cfe3jCVMh8G9E2xkyiBDXV6u4EMzvXCr
2qaSwTV/3upwctcgDIgIXQFGnPNi3lgWkBbKa4oWc55ja+Df4upRr7YXbhpPYGBc

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
bjQ8pNLRSO6zpJLvY5wc5iexS6bgvzfsCWOYkfGuZ0kqJKErSK00f1PZbmYITPv/
IwQIdrOeosuuSRV9Q4uO5KwGEiHn3hjS+uj6b2kvXdXXdYci9CKbvdB7DYcbVjAL
mE4B/c72E04t/4ybB618meH+YsoqPQyOBb+yRC9htBg=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=15760)
`pragma protect data_block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`pragma protect end_protected

endmodule
