//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////


//This module is modification of the axis_buffer from lib_axis. The idea is that this module is parameterized such that sensor data can stream out at
//a constant rate once started. The modifications to support TX sensor data are as follows:
// 1) Remove the WAIT2SEND functionality. Sensor data is streaming and not packet based at this level.
// 2) clear_to_send logic was changed to wait for the ready input to the module. Otherwise, prefetching of data could result in a non-constant tvalid output.
// 3) gbx_axis_tready tied to 1'b1. No backpressure in this module since the design must ingest TX sensor data as it comes in. 
// 4) FIFO status signals all connected as output at top of module.

// Original axis_buffer description follows:
//This module takes AXIS packets on the input and provides a buffer to store the packets. These packets can be
//output from the buffer in a couple different ways. One way is just to output data as soon as it's in the buffer.
//The other mode is to wait until packets are completely in the buffer before sending out. This module assumes a CDC between input and output. 
//A DC FIFO is used at the input side to accomplish this. The output mode to choose depends on the interface the output is connected to. If the downstream
//interface can tolerate the output data valid to deassert during a packet transmission then the output mode can be set
//to output data immediately. If the downstream interface cannot tolerate the data valid deasserting during a transmission
//then the WAIT2SEND mode should be chosen. 

// Additionally, the input and output data widths can be different. A gearbox is used to align the input width to a central width.
// The centeral width is defined as the greater value between input and output data width. If the input data width is less than, then the
// input gear box does the alignment before going into the DC FIFO. This is to prevent any backpressure coming from the gearbox.
// When the input data width is greater than, the gearbox happens afterwards, and the DC FIFO aborbs the backpressure so the output data rate
// is constant. A AXIS reg is also added to help timing with the dc fifo, as well as fix the "cannot wait for ready" requirement the dc fifo logic 
// implements incorrectly.

// Two gearboxes are instantiated, but only one should be adding logic (the other will be din==dout and will just be directly connected wires)

module axis_buffer_tx #(
                    parameter IN_DWIDTH  = 64,
                    parameter OUT_DWIDTH = 8,
                    parameter BUF_DEPTH = 256,
                    parameter DUAL_CLOCK= 0,
                    parameter W_USER = 1,
                    localparam FIFO_DEPTH_W  = ($clog2(BUF_DEPTH))+1
)
(
  input   logic                               in_clk,
  input   logic                               in_rst,
  input   logic                               out_clk,
  input   logic                               out_rst,
                                              
  input   logic                               i_axis_rx_tvalid,
  input   logic   [IN_DWIDTH-1:0]             i_axis_rx_tdata,
  input   logic                               i_axis_rx_tlast,
  input   logic   [W_USER-1:0]                i_axis_rx_tuser,
  input   logic   [(IN_DWIDTH/8)-1:0]         i_axis_rx_tkeep,
  output  logic                               o_axis_rx_tready,
  
  output  logic                               o_axis_tx_tvalid, 
  output  logic   [OUT_DWIDTH-1:0]            o_axis_tx_tdata, 
  output  logic                               o_axis_tx_tlast, 
  output  logic   [W_USER-1:0]                o_axis_tx_tuser, 
  output  logic   [(OUT_DWIDTH/8)-1:0]        o_axis_tx_tkeep, 
  input   logic                               i_axis_tx_tready,
  
  output  logic                               o_buffer_afull,
  output  logic                               o_buffer_aempty,
  output  logic                               o_buffer_full,
  output  logic                               o_buffer_empty,
  output  logic   [FIFO_DEPTH_W-1:0]          o_buffer_wrcnt,
  output  logic   [FIFO_DEPTH_W-1:0]          o_buffer_rdcnt,
  output  logic                               o_buffer_over,
  output  logic                               o_buffer_under
);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
XqykFso37a34AH07/XX6omAYOtNkB3ndGHjwrYKNYOuw0/BqtNMnVFvXQnSKT+zK
XwQOUf0oFAqi+h+jhBwbBReguHMCEZQ+yUVBowCzv1cS9oCAAPK7oOuFbWYhXbU3
OALxfCGXPd0BaTP4WLLtcaPkgTTPDNjhWguaZg+16vsMb2oiC6V7gKOVwS+2mRSW
sYndlFX0GbLQMTwIua7v//cTttvfuuD2W51iQNE4fwhKVJt2G88WS58CeJRmEZaH
s960IXJzvqC+G6kcoDfjBtA5OUefx1cTZYjy6dKDqOJHvQEVAkPoa5UM+V2D8Auh
UCHHIunB/IyC5zfVCVZqaw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
ORtQ6wzQM9r1af2rwiRsmbQH5SsETsbYKDDBCw/JQmAjUCgayvu8+3sYifh2mth2
bQkqlTJjcStwh+LnqcgksYenc/U8QGwClWIN5QoIbq2MM6oy7SZ/isVJbViUgKF2
BEdaCS8KKDpDVKz2Js9dHqVB23x72k/oNve2iZthvNw=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
NhTkXTJulC65XvEUiBkT7lSxWXPWm4EUFRJyUVEntIsn5dBDlDgZrK9F+C6l/pY3
nzIaw2LPjuTgP+bTFQ6rUfj+Zwj6lv9Eal04kWCuNY7QwNeNImO+aUMkUVyLp2a7
tQTVMC8Lepy42GCBcdxfECSUGnuPABQdOuTDkhOyKR24T8/b1DZoyIbT440dxpAP
kYfNPJUiSGYiOxaesXNHqIah+3IXmnyWv8a/YAaseluUHXRQTHg9eEvcukfozB72
KWvXXki2hf7AjQ41KYE1D1XIUJAk1jC++kMn9n8ECTS9QiqEssW9kokJ7qvBmIOI
RINqVF/m7p0GFF4yLpzvbQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
TIYyWDwEMnwEKf5EknvTMqOTDeM489C7GT6Bg1vljO5wonCm90ultYChyWBxF2BO
8QWp/FAWGy7N1E1N5ilDnK6yqeGJVco/2VikzrXfNy57RwVnFdnJKyP7t3eJ4wUr
ewHN3L/QnqSOuDIQKk6OJkgb8AMEBXUOklzU/yhf4cHeAp6HErFSoALri+QZPV/r
9je3CTJRfZr7f0V7Jnh2EuNQpMi11PyIEXruv4x2PWqv6D+QJ0sUi6Iag84mPYB0
2MtVnAP7q7aKpAJkt48KsViA8N5NtzMgLDBDujeGG8oyXH0EFPQ7xgppPmyH524x
ARAWyeJ5lAuFDENetovrxQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
rUyl2GRTHhTNfCwjaUsRY+FZvcHsOajXkAeZ2qPh/jWetfe60FumYWnZ5CVUJgDE
IZ/CyspdshxRgfCKtU7V43QhoPAnq1t7MPuKM+6Cfh784Fp9nctKiM130pZJA0JK
Cwoy9Qm95vqVYhTEg725AsN9LHIomOueQFI1BGSV2ZIMWGNKrOCWVzdGhJaqjjL+
Kwxc03v/6k6B6ivv8szvtPUY7Iybp84PlL5wSC3IQGNDbN9ifNLOeCbU6LVzvJCQ
bQEMfoHfRnT/wlmWHlOF2FBpfrTU62gNOKUNEA1OPP929zuYowpovp9RCn6Awggc
BE+5PB3Pzv/OTZVv74TsHQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
n55K26cjiHsmKAq2SNzlkc3Io5DoyHr1y9N9T3lt6EysIBDX7kqlRRDjh8jPzDhs
lC9AlYXJW5Kn4RF1+hV/9pE0P1NVS7dQgJH2fzsrERSgL94hQUr8CfrQT8ZHOZoz
dYGlruHIuisQpilqedEfnnstqV54+YpXUPsQd9frV86guSaeRbfzg21vexht9sFS
VceCGkytCVwvVpOcV5lwhsP8F4tMvydWT3nJhWGC/OWWoJYki0Grt/flVcv1wEj+
G+ODJEa2Q9ji0J2vvrW2Ny2NLDXQuvcLRQo62M8TIWB5lSE8wexQcKgtYHAzmlus
3HCJMzNCSHRuZTdu87pT/Q==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
pCzQu3yF0N/c9B9aIDvCCIXdUt+jJL8Ftpcr3D22cFh23XVu5a9NsqoqUsbE6jxB
PMx5Rhw9Ls31JAs9L8S2D4ZsPozExkPnLQyviaW4UU6HRZtlVXBkvv7tFkJvcykI
kw+I1lQtofPCZOX24StvQijt8FodQ+CpFCFcCurQQhMmuaQZzba99vXM2vsZpRux
rSzGzB5Er6hdcZzGY1Lhzzpr4CoV9Wp83hLEyANPfNZxOrJ6CWmD784eWaHUTmLn
7VQspVDF4fDi5yWbnvIKdJ6lNa580hMuI0x2STsd+0eatHNK8zkjmfBwLFlDFL0E
kd/HC/642ymrWQNUC+tLPHkMlfJFWqd6aEDvZEYs4BtmWIPB31TAIz7gevSb4IV1
9l0ln3evzM3jl32hDlVZ8jqdKEezLuB8SLXzhMvq/4sRqs5x7qdNZtyh93yKsv3g
uXDcLf7ZPQ2X2k4B8soXIhHrF+YdC/ZBWofyKKzJHZGObhkGC5BA1QRMi3ZuEzLS
t07PLIxi2pCOfhFKOZls62WsKLf+CkMJBo2aKZyKVyn79KOVyrrzAdE19+ZWr/Vp
K+G4zDH7tWItjs+hJwILPGG9q6RRp2qxVQinLyCOHKo9JWuK4HCM/emPXYXE4Oq4
P/0YTHdgTDXNbRsLGoIebUZV1jZlo9O2wyU2cXWgbSl7qX8KvEs8zB97/j7dZ4H8
FMtavr+36jhvfvcux4hTYONJfk9d/jz3qHrwvbl/M2T+VEXOhwc0rXbH2mkgAIRC
NqD27kQ1Qk2sjEiO9tQnrztZF+LpEvyAAx9vJhkqAPCIhyGl7BnccNvPdGv0ztvK
3OYpxHCZD9MwyujBryuBVwxYhqXz5Ej3YmoNzVRK+FywuD7KAfF+DaHVwnAxyLeM
PP7S+B2xH0Ej991ohv0zhFkZ7mYjLtKmtAsJugxZs3lLhhq6ARY4TmVYf/ctLcBo
U4TgXc1V7XGG8EaGyH6JBGkYGTn2t1AbN46qGtt9ITglRPyH/n4u6PDTpdrppsCE
Sq6bLjQvFVk+cStuF4M0Zdv/Wg6Qe2oX6meHuphYs7h5jPXcCfXGpLlMuNi4Sgy9
bQ+RNI9hTo1K5pvHoyy7iiD7dbKefG7wcqVrxV6LSQ4dOkjuITa6fE3OBzj3LkT5
FZxU9hmkzXkhQ5u9xUcAdSYlals3q5gGx6ZV52nimCbP75Yvnv5jZDwSlcuKdLHa
TgIGHwcuGie1nR0dPQfgPPAD1y6DHdsgcDU5Mwbv9tMAbpSA6uSqW9nF5Ta1+r0r

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
jad2LHzNdcPjeQOXosj+zNbvmmuZgE+Ih4DmPfw124yoVkXzEfG/PgpZYwiv3DK1
qQbbg0+ITm/hqbnJALEmC4XbSK4y9zp6MyNcFoVwBI4vdx1z0nrWpms3z37u54PA
uzs/apetMPd9L1KXTEMi0dFXibmx/9pbNrSEpMPBOD8=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=12304)
`pragma protect data_block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`pragma protect end_protected

endmodule
