//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////


module ptp_top
  import ptp_pkg::*;
  import apb_pkg::*;
  import regmap_pkg::*;
#(
  parameter HIF_CLK_FREQ = 156250000, // clock period in Hz
  parameter PTP_CLK_FREQ = 100000000, // clock period in Hz
  parameter AXI_DWIDTH   = 64,
  parameter NUM_HOST     = 2,
  parameter W_NUM_HOST = NUM_HOST>1 ? $clog2(NUM_HOST) : 1,
  parameter KEEP_WIDTH   = AXI_DWIDTH/8
)(
  input                   i_pclk,
  input                   i_prst,

  input                   i_apb_clk,
  input                   i_apb_rst,
  input  apb_m2s          i_apb_m2s,
  output apb_s2m          o_apb_s2m,

  ///////////////////////////////////////////////////////////
  // AXIS
  ///////////////////////////////////////////////////////////
  input                   i_hif_clk,
  input                   i_hif_rst,
  input  [AXI_DWIDTH-1:0] i_axis_tdata,
  input  [KEEP_WIDTH-1:0] i_axis_tkeep,
  input                   i_axis_tvalid,
  input                   i_axis_tuser,
  input                   i_axis_tlast,
  output                  o_axis_tready,

  output [AXI_DWIDTH-1:0] o_axis_tdata,
  output [KEEP_WIDTH-1:0] o_axis_tkeep,
  output                  o_axis_tvalid,
  output                  o_axis_tuser,
  output                  o_axis_tlast,
  input                   i_axis_tready,

  input  [          47:0] i_fpga_mac_addr,
  input                   i_ptp_tx_ts_vld, //Stamp TX Timestamp valid
  output                  o_ptp_en_sync_rx, //Enable Sync RX
  ////////////////////////////////////////////////////////////
  // PTP
  ////////////////////////////////////////////////////////////
  input                   i_enable,
  output                  o_pps,           // pulse per second
  output [          47:0] o_sec,           // v2 PTP seconds
  output [          31:0] o_nano_sec,      // v2 PTP nano seconds round to the nearest nano second
  output [          47:0] o_frac_nano_sec  // v2 PTP fraction nano seconds
);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
u7Qb/nVokaFVUYIU22uFKQddVKlnmKMGoRvc7SXmSwOtk+UjPHa/WCSyyy9H7X56
0EqYH7q4X1shB7NdrK6i/8efh43N0sUsSJgKHBfvP15Jfn/GL/4ESUUgHFKAojor
somDflzWf38G715k3kR3JKDAB3oEey9UVtfLyo3Zi3Y3loF/PZgdUHzJUZF1AREk
uuWwLJcVQkBuY3FCGASymxu9Hu0wyD2ZijRuGdMTZ6ONa0qaRQv7WS+BWJ2Hc2g/
fVTo7CQfNjxuuWj6jR620F+NsNdaLGPlkkrJ3HB355f4+1CvhiPAI58mUTjp9YA1
fRB/S+i7KvILplCgmy2oXw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
KoRdBUM9pVtv9nCZIh3KPB9qWdt6S9XSTsmYCHbGHPrjB+p2QeCiKCphHPb3zCrr
vAgazv0eJq3IwTB1LVAeJyOF24KyTAZnV85ccgNVaJHpn5i5+RpxbTGOfs7El9Bm
4DaQFA29CdkVxVBirDPc7uChERuT67UzybgoiazwmvI=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
FAg5jzAFHZeNWA37iwKGQKFa/FDptqQ/b5tvSpaEMGPnXSDgG5UepzkjjAgEX5lY
4FXVvWJGDSnTwEEuV+7AIT6NZBiO6adBD+oUEVao8BUFiBrZvs/4qwALYX/fUlge
lTAHD9AJ/a3fwgBDSp/wyCE8dMLVGqFH7XAHwAH2vVwZ9/VDE1lZlTIrgFqsj5vA
bsltpHVl1vaJ4ZuSf/1psr4VMfi1x12OjtsxMGuKhp0IRZcmybj1wL4NSljxYjG1
J0Qf//Wl7QdsHNsUP4e7Tm52cTsBhMb3tDu343F8bfJwUWKd1fHPdE3maEk7hoZ3
BpgWjwJLitIJmHl6X2krKw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
Vw8VkPhW49nbEaefVGEG4+96cmMf1L7ffv6HuOI92v2+BHKUdjjX672KpunBajPD
4+pZN/xRGFKqnUaurQ15n285+XA7rR74AUOoQc0XuYeZ3wNyoG3Nov5dweA76VhL
MWtQzsnAv0UcAeM/EoL6KFaRM2FGQVhV59Y/F68dQaAfAGZkD9EE4wsLcur+sMRQ
/aM5/sCoFDXCg3WgBe5vKoeR1z4D3K5c8JOGEcFe4hUnjefayOxXYyEbnNSDm+9+
nnj6RmRaL5qVQp2WBlxAP28FZGw9HJfOd2sO9xeIv6DpS8FIb10xR2bF+KccmJTP
otMyCORNVudpD+9jCm6Z/A==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
nYVGy5kdsjLKZ2ANH93ZYbRdLjh7y5rg11GuplWG5L//f4W9ZFIx3g399/YCKw/Y
ITmZr9NSnxKACPxeX4m1g5e3Wzagem6wHKzhS+kGc7BE5reTxjxafKXGJmbm16jR
68+I4mBWxQUJQVF3cC3uV2X68FAMUt0n7OurTygve/l2nNorve6jDpzkER1P5UJw
uLTGvRCHSa7CwJCTgq1ztkUAn0UxAzM9yc8Ad+FmhVqIVvyTK3JKjZNGhouxLCe1
fxqrNrcSydbQ15k5nd458q1UZ4anDQbsYF5WbBl6y0nP86flWrzj2HqUGT3oa6vz
QCDFcqhEdYxuhOpO3KBXbQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
DonyDw5JfwLJA5h9OuMWWemX6hItOb4dRAXLxksUl7dCvHjAMAQ8oaKGbIrHh3z5
/cgB3bNSRhFAPwhoeFM+6S+xlTA1v1W9TcXIty7v92fILVXZ1iLkhhWlz606ko6c
k9g34j4MSl9DLbWO9TO01B6QUApjospy1ITWjUeYcgi3WXqLitDv0ssf9zd/1G7Q
us4xVjPWVp5irJfNyl2+MpwLln026dTOctrgG7r4Jfq71RHjIrBhN4BallZaq+c1
HizhKABHYxjdJesdudMKEtJxzD+dET6jVU0kR8Cpj5rAE5ZyyOjJO+bqwlH360+h
ikWix5NcHP0LyDYtBdpRjg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
WyP33gny7oWp6j2tPKHi5c0njMjIOZy0MDbTKv81tJBDrn9V/d8Bn7AE6XSegTjG
ZmeN6Rjk5nkPkU21LsqGrWgtUgjnrTWfETZ3ccGtqgifb1slJWKSTqwfaHRpHyNY
Cd6+KwlkSBMPP87JXUfB6eQu/qM+MWxt61/jD0WAZXVH4HD0nRK6+Zgck/un1CwN
vM3/R6DK8gzyGsDHHkKbGbyEpg1ltzO0xJFMvzTln+x6GSxT9Zz1gw/SNnP6LRdg
6d67UcaG39E2IJZ7QNgShn/Fup8PVMOmF9Gj2n3+JReTxOmw3nriTZM8+OaYFUce
pBZlBng+90IiOu7GvFdLtehDLX8DVyqFsMroPP6y5Mzgp6CSI3a8ZfmOUQFUTR0Z
m4tfHSxt0jVnzoR/kCnT9qfhvMLrBtJgnMureki9M6DhCzqRowAVWO+S29e2L0Ff
d7BW14IqYyUdNPgbkkqAdW5rTsITKVXSAOb2ByKRn+PAatC8Nx5pd4+uumDqVZz5
2kH/9P1prk7kYjC8YA17bmbjw2HvPSCC0B+TxvjJnO+09eirHfrRPZTGhwJRIJiZ
+o5leuCSchIibEyyla1CsZ0U+gOUSa9hOWSE8kLV6KBg2kLqnF7o1jfGEEw3ojmo
kavjivacqziLwjqiiFJMa6gS6YTZAvtqk66WoIRFyFAq6VcSlEV3EWz++idBkUXC
SuXY7kh/AKfHRlQrjrmRGqemRp5cWsWfRrteF/O24AYi0mOHNCMmqHZZkbPgFNX2
5RZOaRr/2ANCq3gcTdXM+z4TVsCwmlME2ShpnfBx2a533Cw1GkEHD7a7+3RGzzpL
YHXSTTlCUYYuEB6E8kHzNR9zVL8+I2DBngreM5F18zHX414UUFxoXIZ2TRS3oJ1b
j0dJU1ndrTIGhVsCrh5qgFY3M2WuYO9pOORkrcDioYtBY0IzZ8D7F0YecbwEXSus
apuANIaSeKo+apa/wldMRp/ZBXD/hnEL7hZC7PkmszVGvq/2MFOc6S7QimuZg98s
amkyR7knTjqDkfTn5YBr7MthvHVvBCzGN4q9zPxZJBPk6nuj9fm2vnhpBfxVNNNk
ZFf9YcZS/w2KbYugSwyxCZ6rOIli+xZSobiNFXvCg+FcOutC78z9R9xQQOyXYFM1
YUk82eY6DIkatEEMunQ0/IP/daPTPqsYb5aoHxFgfbTr50JMlQKn2LlaYQQR67H/
jTEUQ/bIvYLxDHROsR0sTrDVEaa8S/usYUQ20w/9xSO6ZS1aqXFrS92GCWKFuR9X

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
dkyiF3GbLAWVFt35NwN9h9JKshmZq1YqTJVVuYMXZhIcRSdAS6QqXgY8sJcGRVVr
lVMs/kSPCZkDzNedJf+cj4D3FwEX8cwoZoAKkvRpOAfBWRWyNkdfK2CWTfsoWbob
971N3ZGPyMyspyEcMPGRAeaXWzTQzJH+bpDRH6SHDkA=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=17632)
`pragma protect data_block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`pragma protect end_protected

endmodule

