//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////


module ptp_dly
  import ptp_pkg::*;
#(
  parameter PTP_EGRESS_WIDTH  = 64*8,
  parameter NUM_HOST          = 2,
  parameter W_NUM_HOST        = NUM_HOST>1 ? $clog2(NUM_HOST) : 1
)(
  input                           i_pclk,
  input                           i_prst,

  input                           i_hif_clk,
  input                           i_hif_rst,
  
  input  [47:0]                   i_fpga_mac_addr ,
  output [PTP_EGRESS_WIDTH-1:0]   o_ptp_tx_data,
  output                          o_ptp_tx_vld,
  output [$clog2(PTP_EGRESS_WIDTH/8)-1:0] o_ptp_tx_len,

  input                           i_is_1588_e2e,
  input                           i_is_1588_p2p,
  input                           i_is_gPTP,
  input                           i_dly_resp_msg_vld,
  input                           i_pdly_req_msg_vld,
  input                           i_pdly_resp_msg_vld,
  input                           i_pdly_resp_follow_up_msg_vld,

  input                           i_two_step,
  input  [47:0]                   i_cf_ns,
  input  [79:0]                   i_timestamp,
  input  [15:0]                   i_seq_id,
  input  [63:0]                   i_src_clkid,
  input  [15:0]                   i_src_portid,
  input  [63:0]                   i_req_src_portidentity,
  input  [15:0]                   i_req_src_portid,

  input                           i_dly_req,
  input                           i_pps,
  input  [47:0]                   i_sec,
  input  [31:0]                   i_nano_sec,
  input                           i_tx_ts_vld,

  output [79:0]                   o_dly_t1_ts,
  output [79:0]                   o_dly_t2_ts,
  output [79:0]                   o_dly_t3_ts,
  output [79:0]                   o_dly_t4_ts,
  output [47:0]                   o_dly_cf_ns,
  output                          o_dly_ld
);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
ayg8l9sljULTun0ukV5z0yW6qyBTXkVIsNp0DXF6Qu/OZZ4MWARzA8VeaV92xs9Z
SCN89fQka0aCym7UjrzjVlkWItd87C4Ggi8OjZI08jpE3fwyxeTUYkE3V3/HDlai
ji5JGw/8xcsagyunqsQYUsXbErZ3bJe0btWXGujSs2aZsKKW0MDYredx8f9EWPiJ
gCuC/DpyfQPjT8fJ1tmLCEl8k3xmT+slkIDwX5D0kz/RY7Gm8lpyHzcryTjmfwsZ
c1y/RX3KnUGDA7YEopTGSTAzyinjd6v9cMejfa/Y8S6+XJKy8U34VhFoVsZzRCKG
DJw4dpo1dIgP89ucAFORUg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
ysP6eT5/CerRWxCQYhjYcwoHrUFL1dAflNmi6ghiBkwR4b9heUB4DZ7+K5NF2Iuw
3R+PwO17VdD/vmOW6LZXWEZ8ua5yizgu3DSWgV64KmusYONl7Ov2l+O3FLoB9gC/
+kqHZnCiHwcpyBOQ/p+qopRgfUl7o20Q0MkMjo+TMmY=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
i2gwNTZ7coqgH2LJf8/ADrU2+YOivxZaoyIsuAzqPllvMixU6GjLeP8ES/EMhPzo
pg5+kJFt0yqVF7uJUIkUR9LPbbXQne0O0Ea+5zebHfaZt5tOO86r4HrYAKo7VoDv
S0d0b70Hqa4jUVYXO6q5jKc33q/GDQBxDs8XB91E8XWj4hpiVazRIez2Aj3q27rr
S4Ae+pFUMNLB6IqJ69b2ht4kVg5axlMaeVhGiAOT5pxabmo1pxNFwF0m1C1Cxi1B
0kiIFuHfEKAjDQRtqNzS3wPbgKSd4Gj7GowZTDrv2rhoW4/dpbbBRkDC37erHuar
Kt/wBmjepfh9ya4b7fbmNQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
XYJRJ+GDf5JTAW3rpz6a8dfi4W34woC+PN++NXxRtvqLtHvB66B9Uos9cL0Lu3ws
OC62C5fcePae7vfkk1M51K4f2bDAukGvA9rCINqCY5y84RuYxBnVtFP4cTVJIYlr
1cQCrF0FK1VMCcJQ1GjLPlg6b/1zXztZF9k3PIexTaiHQ88JutmKUYvKyRO4dPCf
guWYs8vXnxJWFL4gSnjcSIaUr46J8/w2yOlXJerD0tEbJxKTO0nzf3oEePh0IRBy
jCAjmgRLvuuJkxKTUtEpZ6/RF50eXunasT5zMZaf78yEzJHJuaLUN2CkqiVrz02d
+Dj0XYGodbwzyDoRLGgVyg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
v5lmLvZZ1Uz/t1iyNlapKxOQjOS9SC4TSXt6JWXtohD6ZVKUmOCumOvNEnvH96Le
A+yUhux3aIA+ugarMsMAORIcWMEm+7E6qbWg8v6ARmXCxNsnuvbo87Scs87ecPEB
s0qYh1smTAQ+Wk9rNtMMe+bufnShu4Y2PHr5FeNPvbpfqEgvyf/sojEYsbtnaIgy
7sqg7mjO5gL5mOd8tN4hhQUwzX9Jc7cPfwwNYTFUGPWq10Lo2ZSvxid6QqzK7rL8
N6FDiLVgCHa+mDPDH2p1QnrF3XbfCMWEVpUFrDV8C2H2ChTJkKu6YCSeNNsE4iFb
XgAGbSN0TFjUxcO6n5NAaQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
aaMwR+WU6rum5F5d3nFcr+eOz/cqcY9M9AlLBKPMQPBfXyFeIvWWtJaXzoqnUQRH
r5j0uBG6LpzKweirAofT5UTo3AnhjEZ91vgWOZb0bUviMHukBbfP0rMLE4O4xOV1
0xdt68RaB9OCuLJwXA6pAHFCaC5xCTYtV2UBFZZJorv1ZslT3Ggoyw93wMX43IWy
9lMN9CFnrti/I2hfL9HL+tMEpPcY5qULK47R2ray6AxWXpUrT3VdMSpdBBj/AYxz
N4PIVYYkYsXnzgwvVc2IAh6kv9j0HwOE+f9jN+IkUbKolcPVNZ0f3SsdxU8TCecp
093fdTHQcIgTCEabXQn0DA==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
ZGDqIMMvGDUPi7P5WAkPkEj1dECh824svgv+bSlzwbVjMyoueY5JHyn3x5tSue6p
OofWlz7YS87nY/6YRnbB9BS0P2rRTrFMmZBQBH1Ci/rPsz2f4F0t1UhJ/uTzTs1H
AGyKNTlviAQZdGX/ZC3YYtO/+/x1lzUuwRW6YSk/gDB2lIVMrupT8jjW/2p3/fJV
O4+IR7xzJQnkK3EQE2disJA7mA1umaNb8TXIRTacjkk11EloILOwkJ5QFaElwEZw
dtTUl3MDFJwiCm2TKVGqCOzWN/Ff0Pdk1N4BPzqQMnJEFjLC58uoOBQYl4A3KgSh
ez8UmyNjNCjcZt2TmxnzQh1p/Ep5JqSg9bRpSBhjsRiGRDGFj7b+GXzclfoW2pmt
6knf3qy1GD8gugoqfyxr/0LbxIiZGVT+pgR8OwR2djvLd9GMrfIMZYcyujeFotWH
0jh4bN4PSNWyFVK31BQOCTZ9NoX0HBSOC816FjwFI2me+YLNhRDF+SFK7+vk9OBy
fG7YeeXOHEdU8JEuuNmuHxvyhWKaU2GBqvago2OxRo+NLvvuxbZvm4yBcLwf4dOk
8lvT5qAhTMHGBHiwEWjxLTDNeYHu0619pXKKTVnfVD2kU7xbaPFr/T0IOgPFk/RO
Y/G8p51sOsY7MKUNDCkSAuQ3suMKS6s9m383dBp7SsIIwloEQkDyd/XyxN3LZMOC
vTrDN0q+bbIt4GGMDPh4FuVqw2qlHua7kkjoHD293V6Zt5IToP59KHL21crkOJr4
j5QyJuYwzwJHdn2nheRGrgfEDgQUKTok0IC0NWu2zxoXRKJcE9sqMcNcxT7cn7P2
zMUsMuvNnkNo/Pl0e5F5XtR7yBsj4zES0ykZ47HFFfZCY1j0Yf7c4RUfxBLy+USj
KWMFqPl/l7Vt6KlS8m6xICKMTr0p6hWeSZU59Y4oenSKfzL/g44UF59s3x47/ivE
lyfikkHD0Hz0Bndi3eiyAwPwCUe1kHuP6r+IT6ajdy7zl7n2cZI5pLcHAhNqbSEp
hwLYPlloOtDRT2RTzX3kR9VcxMWDMpTGgKp4DUXJT44vu5JeZR/yL3BSl0Tb1eQ8
5SzGRa2Hz+97ewGObfYrEbSG8Md30gPa9PvkJQ4MQPw0QBjehF3t9QOKqAKImWjn
xCcsbKphOJQDx0cMaGcU1QoILN12enznxIU0ObLryToX82Dm25PwIODCHGicKLs2
EAR00BPqp/B0ei3U0efBw0fE3hIWe9wORhwUIK3/NNeVxBKmqWwG/+Q/BUzj9+oA

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
miccT2lJ8/0Y/r8I+d3HgT0crNGihIIXTIqBD9KD3nGcW0tuGayNEN+M4GstC7WH
hTj2Ciee2DfaCWm4BVAwebcuQnThzyZlKvYqHosXEK/h+CyTc4zbIk8CNDVg35pc
bkXC6/G+LjQwZjVk1fS5lxIO9kO+oKxwNnBmZhIQOBc=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=11456)
`pragma protect data_block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`pragma protect end_protected

endmodule
