//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////



module packetizer # ( 
    parameter DIN_WIDTH = 64,
    parameter SORT_ENABLE = 1,
    parameter SORT_RESOLUTION = 2,
    parameter PIPE_DEPTH = 3,
    parameter VP_COUNT = 1,
    parameter VP_SIZE = 32,
    parameter DOUT_WIDTH = 64,
    parameter LUT_RAM = 1,
    parameter NUM_CYCLES = 3,
    parameter DYNAMIC_VP  = 1,
    parameter MIXED_VP_SIZE = 1,
    parameter MTU = 1500,
    parameter W_USER = 1,
    localparam RAM_D_WIDTH = 32,
    localparam DOUT_KWIDTH = DOUT_WIDTH/8
) (
    // System
    input                                            i_sclk,
    input                                            i_srst,
    // System
    input                                            i_pclk,
    input                                            i_prst,
    // RAM Access
    input                                            ram_wr,
    input  [15:0]                                    ram_addr,
    input  [RAM_D_WIDTH-1:0]                         ram_wr_data,
    output  [VP_COUNT-1:0]                           vp_is_empty,
    output  [VP_COUNT-1:0]                           vp_almost_full,
    // Data In
    input  [DIN_WIDTH-1:0]                           din,
    input                                            din_valid,
    input   [W_USER-1:0]                             din_tuser,
    input                                            din_wait,
    input                                            din_stall,
    input                                            din_tlast,
    input                                            duplicate,
    // Data Output
    output  [VP_COUNT-1:0]                           o_axis_tvalid,
    output  [VP_COUNT-1:0]                           o_axis_tlast,
    output  [DOUT_WIDTH-1:0]                         o_axis_tdata [VP_COUNT-1:0],
    output  [DOUT_KWIDTH-1:0]                        o_axis_tkeep [VP_COUNT-1:0],
    output  [VP_COUNT-1:0]                           o_axis_tuser,
    input   [VP_COUNT-1:0]                           i_axis_tready
    );





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
UzBkT/IkPOyRtqBemAaUBS9UqLXQ5YKkL1rt3LPxLC7M2vtzfJQQ4YxFGGh0K3rI
G+yG6Zj+sEYZBYY2oqFfBGBPMFHaAwTVLAAtDIwYA0gEhSOVDYBS9fciZak9IztF
3wDFgMoK0ORx3vdGjuCeUBFoEOcJoTmlYjy1QHUl1rZcfcXv48+htpQQ7Hn8V6R6
4qdseT30cFh1MRehYP5jn11L/tZ2dUmXo60FS4ZKmY86/cakS9nZd76FK8iKjner
6JhoigQ9lhnV53Ovzq6O+HdHnKrcmO7heMWI/9TWQdhRGHN8PxmR9CPWki8feDnW
P8VR57rf/11yCzwQ/Dlmjg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
3icNUWao+Pc0x1JcFzb9K0JRX8A7cXbPAMP5pwwODyWZ8+ZM8zPhIT2zNRtWH43c
8Ax/MbLKU+BLPRjT0zbBWZPQcPB2o5SSi1AWGWw6qhB4L4xCd3rFOUqZZ5o/VfLM
fmeF+km5NwNfMODDlld7K9auGXGWAaYuUu1V4Eux2dw=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
TqE8gzt1ZUYOXDMcEMz/2cf/oUDAnwH1ILwYFm430wwASeEVD0/rw+PVHdZH5oug
YUbOEvCtQt/BAogwtrhz/N0KBBLgzYE4ns6lHP6HCqxG508d3ydRHe53q4Kd0+DY
FLQV8WI8GuSFdadfZvUfMHCqunDBLNznBtozW/J6wi55ghoh3hIkvHqxX3R0NqAK
PYG6aAj8nAo0p8TC/WsWfQTNBXm5Xz/XSvNbroA7LfFhmW+0MnL/7qgH99FT6ATw
VJQ9IAlSAQStuv3W5NDlut+Ip9hxw7tXAGQS74//vNyRPAd7Em/W+U0YFW6X3HYC
0fryay/Dl/y+xY5qnD5EQA==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
gTHs1Hx80ekOocPZqBEC9HE53czClEqOouJO0f0U5a3LHPC6mZc0Pw1lFuxQIAQw
z9ilVnzF3hKlP9EFrPegsDPJXw25+GG/IvcLnCSXxey0gwmYbm6tCrX4wtgjd97a
soHuWu4At+e5nwPz7kAzL0iU2kZKMg0bWVq19NZsApX/psGdoQx1RZwUi+6tAqXC
o8b59gTq0i1Pb0TK73IcHxrIpGW25OHpMtNQGyXAX6jrCyC1qAtC9yVFFQY3v5qe
ZGHo8J+mSgzlsmY65aI7VnirrohiOn1vikosK1HNFcp3doF9O/BJvQF9HABQig0s
Fd6OCepdUX74r2yKmbZahQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
PbS5yAkkFruMcldRsc7D32sheJ6RaRT7xAAE2rNP/XCuPZqw2QwPbKnpIOmWj/Dl
j+yr6vaLemMcj9ivrjA2Pyc7mi6ko6v3/4yzQdkTMrTOd5X3zBaf9lDh/4a7OE5W
o6Y7ICH5sEfyszE9gA7BsJmOi05IZJrxfq34YSDk2TSOHS0EirwCwHsY7ewTvEqN
iEhIQQzpyJCl1P/uQdTMz0PQRlpAvVxeyBkA3ZSsXyPd91sLjAJWj3+oS0V9GyCX
qYGAmmqUhjp9j97FG5QxbFk/2VjuszcBLz10dDXsos4mUfK+Au9oxJnX35LqVUm4
mi510OFN9bOjt1RskSvP0w==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
mBkc10yW5I4cnr61zW4PL9AYTIHBiGAOe2flgPz/oNA4Oot/iJ9rWbfGlmDVF7vu
dV1vIQ4rlmWXFtVBJ/FOXy/UYSgDk03aUHrf9CLcZVN5g02s0UZm0Z6bFFEhpk4C
5YExzmZy3ncY6xoRJmF9g+FZLE1ThnPdix09gCRxbHemxQ4U4CRLHOXgAEHCknoD
3cmTjb4JKfe0wGtTxhZW2Q4M3JHeWfjayOjgPL+s3pPtUUCrOgVXXuaTL7hmGeJA
ZJ15KK6T5LYiCco4fDXJ4utxn0nyauF1t9XvG9TpyYra2AbTJjGha5AaRtyE0EV/
2+1U59HE7ZSQaBxLiHGjBQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
uZmc6UDvLyabIFPaPxmch619lzDsDiziQTqSKrUJsEjh4XK2m+DLdcFgmide8fos
BuOr5Wu07c27nTKHn1dcJohIQVZpIF8ZTzPV7sE+4MKE4rWzMBfhZ4OvgzF+jSjU
jwVKBI+UrX1Bz+rm1fQIEqhhe3MzVfvY/4wUmrh/qbqC4nALupjkziopxPkMqxW6
snDusRBjoelMpJkPopjUWW9b+0AQeCLMuTPN32yhrs7MHYR7fZc/3ervCyixyX4t
+0I6SSP8peUGoK6YE7UiYPB/GmbYhPXtZK0aj7Xd+U8DbTZWQhVkP4JS2j4lTKL3
6NMgKfumX9g4dQzG4/HaesXk0HZ789Zsc26LHpz8C+OIPigON/tXiI+1sXLL5VDz
HTWDITsZPAPWh7TAicSZX3tfCjySxU2yyTHShlQQlqC4hGPnr317lPVlUMQ+b85r
Z1tn5FNV6Q3Q0upN9d2L/NTKkPElv8iptx0SZNH7QMPOVjNu78MdRyIuCaclDIox
mLhs/yfVe9y9KvG97TT0DkSriWk0p1ofXlnHj1jzdnY5AxFm24iNm/cT6Kev5w2h
4XGc09pePn2ZLKQ3iJSJMjdtn3aPeZqP/aOaNkDz9N5HSOwkf9XoD6ejL9dkpzpj
JqfpdWfefU4RZQTlcIsQmzKgMtBu+nJFxXdOYqiYArVVQi5XMTDHFPQpcqaJhpr3
ifsHoMXbI8YddVFY28bTzQaGdYwCB1uAc+0T6+NhovVeLAAcbG5AYZ6bTiWbmlQl
7yIsSR7wjjZVWg3Ymngra4qqk+DxG3AfqABv4uH6PaPmQ+Z181j3XQ2GRSloNPJU
hXBEubBMDP67hu/M89pQP8G12JGMzl2nk2+qowYuRCbx/BNhSsPw9M+YMYMX1/md
6XQEpsYMNzPcG0LtAZ2Z5v1dtoEqgVF8Ph63614JU5YZZH61BPLj4nZfqRsDHZTC
g+iFVMLT5Z/8dxz3VjSApaOrWyIRGuECbIgif1mZlt44BYpwPr2BNsz5qW6WDyih
MYg6YBsLQZmveeKcUQiLzjN1mDSjtjF43TwtyutDN7cZzSR3mNigI7qEWg7gmWMu
6v1pycy+4X86Wozt+rizHCxigE1BPFors6JWB7eRYY/5k0JyYi0WdW1KD+EtozdV
L9QXnQrDhNfGEB/8pWEDxwlaSeaECgksB9aOIfvUyiSKWUrNxiKzTZIdT/H1xR4o
RuFCRbcrM+3TKQK7TbaAA9/KF9TpXGdr9VKtJbTVMH7XLFcOZmi6teG3UNGaS3IZ

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
Hyh5XSFATXKgjCYHSSacAbOQRTdv1ZuIH//SU0RxkgbvLupUuoTChMpkuK0uHfbt
PKjogcNeyuqylS7oKqUlOsNxcnDQCjuL625rG3OhznAyMKaJvD91bAvQpPdAR88R
iv1LeIrEJKGmVIMcRzp0KkXTzUdSnFw9lo9Cin+tVFs=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=11152)
`pragma protect data_block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`pragma protect end_protected

endmodule
