//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////


// MIXED WIDTH Buffer, DOUT/DIN must be an integer
module axis_gearbox #(
  parameter DIN_WIDTH   = 8,                // bit width of input data
  parameter DOUT_WIDTH  = 32,              // bit width of output data
  parameter W_USER      = 1,
  localparam EVEN_MULTIPLE = (((DIN_WIDTH%DOUT_WIDTH) == '0) || ((DOUT_WIDTH%DIN_WIDTH) == '0)),
  localparam IN_W_KEEP     = DIN_WIDTH/8,
  localparam OUT_W_KEEP    = DOUT_WIDTH/8,
  localparam IN_W_USER     = W_USER,
  localparam OUT_W_USER    = (EVEN_MULTIPLE) ? W_USER : (IN_W_USER * DOUT_WIDTH / DIN_WIDTH)
)(

  input                                       clk,
  input                                       rst,

  input                                       i_axis_rx_tvalid,
  input           [DIN_WIDTH-1:0]             i_axis_rx_tdata,
  input                                       i_axis_rx_tlast,
  input           [IN_W_USER-1:0]             i_axis_rx_tuser,
  input           [IN_W_KEEP-1:0]             i_axis_rx_tkeep,
  output                                      o_axis_rx_tready,
  
  output                                      o_axis_tx_tvalid, 
  output          [DOUT_WIDTH-1:0]            o_axis_tx_tdata, 
  output                                      o_axis_tx_tlast, 
  output          [OUT_W_USER-1:0]            o_axis_tx_tuser, 
  output          [OUT_W_KEEP-1:0]            o_axis_tx_tkeep, 
  input                                       i_axis_tx_tready
);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
W3nTbfPDvpUOqQVNpBlsdH3tRsXHhPrXauF7ms8+8Yllj0+ZaJT++FuUoVpd2q3w
0YE81o/W3eXp+ESi13RfYQwrqOqJaIkPA0We2BcX4FtkCh0nh3VIgUvLdjT5FSU6
H7T7xxpJawQEyWousv0PEU3fDjZ4kxTF5iuSH5pUsserDfw9wRmf2PrcndlEHKEk
Rge3t3KJ3poW5DanGWz9FeVtnLNR6eLNrWqYQqPzRVLoMIBv2TNsS79b27+xd46k
+/bkRrk2b+qtv5LH7A6dJwyUjv8keSgW8/M/l6DUXfY6bxdCUIjRaHazw3NTw1xI
fKwjVaeKW3KIOHPxNV1Qcg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
CtOTBxPxG/NLbAOYi/7t6Liitj1QclurOu/6s2Co88ulr3W1L2ckcnnq2cyy6+Vv
lawgGnX+QLud6kl3RrG3yZmNSB/UkpSEkTAD+TzwNtLIAZlq8JwhH2MMMUTPGh4G
UGl4gPkX52rDdb+u1+NvRcYEugf1EvLHriPU4zSfLc0=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
gnw7nK6EuwmbhKcG5X9J2/uC+GCioOyRAEAP/7x/g76sGaSEax17ODUArV0Bqr2W
ymHZhITdIrfJse2fAOWr2rNlwyp6WVeSVoC+kbAZit3cc/Ml1U/dr/b09GiDUOu2
TfhXfxaWuga8EHmzcaGE1vMTEPpJWwPvttoLFPY70AAwMiTxB2/pJAQhW6VARfRk
IA63Nw34oBkc4EMYY16d4iWHSSOXmP+6RIwb5xTkYcvLMmwaOJsrHWb0vhNdCau4
MikBxdZt8Saix4GuBZ31Qf5rkjkzDcgXAK3eNhvaFgnna44+orWYsYBFRdUA1lM/
Y0whaDy7R18XgmeDqvv33g==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
TFvkpeVKMVGsZcqPDfuKOhSIz7+AprhoIbsWRGcsYwbJxX+LoDe2PfXoqWRLXA2Q
zOel/8JqiSereAMqZ9/yNihUj/jslr8lRmsODoyc9CK3A/qwF64yrZgnFX85u2yg
C1wXL19147QUJQdMhuOkvJyvo05bTZRRAqsB0YPISVC1WCrbrdV7gSBjcduhUif0
8MVCbjsst+JCpK867X5hE8iP+bA7H7dN/SqO0ou0DBZhFEiAMi2ounGcgqxfAtcj
4mvRpKkWfPNAZo/5Tvp48uGMSvp872WFZ7x15E3axgPTUS1CI5rUMFaczCJtQqpD
VoL6IH+wjj/Ij6lL/8BbOw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
p2GRbf9wJehVpwL9Olbzh+XHQkTavG54CogBM2PKk6JsgCaKdBpa0fzpuHzB3jig
0AzgKCTzOl84kTESKHR1hI3dPH186mQ1OXfPcf0qWA2y5HuagGvOBFxLHBQ2x2ix
aTKPbVIp7bATgxUKBaBSYrJFe1SR5SPwNWsQsEI4BEszsc3JqP4wiwehz+9HW0qZ
KDZHaXrB62cZehxJZX+8VuFRm0sIi+T6m/bv7IPNnz2gfeOVAgVQFBGtibZS3bt9
G5J9pSPImJ6XD1e3zo8b6HORXwRhZRLH1r5fZRgxcHDval0MLyJezXdgUpw4/7Vb
RcL9/A6WM/glPSo+a72wRA==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
dXQwtNQMc9eaBOjoOv8SPUZDuuVw050IeTpKRFXZ0S8c3WcU+eDfjh8hgDYizHcN
CWvD+anHy3GUeFCFzC8YeVZbZJE+Nq6N0SLIRSB1SGdLKLFxPuBNQM8tQ/cfsCL/
BlvuuQtWbaPtk31tEHSiSgLKvHo+KUJnTe9geF5L/mWU5fXxOWU4S6q5nY0ngOEs
K0i5u9wc/hkk5IAMO8cML1lUbd4ejqBrM4aeCIottVA2fvUDBqhCsGUm381Vr9rv
Z3jHU7eOL2DhCN/AK3qudKCAYWVM5gUJZB/zkrU1wTMtHYlXuzd7EzkPxr2ptlEJ
IkAHUBF0pC8IRa3uerULXQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
xkm1DdxeoPuhPUyTTD73Qp/FOIOkXwhtzTFLVECcr/zJPIfTBLbHXyhIX6yXAdY8
f5E7i/cURik34/fr3428TiOQ6QWw3lO8/oPFl6wrhDR1WFcceu52UcIizqeXR07b
ZGvfOIz+avZcQjJFgcp0HLzTPk80RCNjBqW57F2sU2oXpwayu0+jD0Ssv3Tgpe5N
P9vM+Mto7Zzaa3iM+D3aDr329q7XLQo1KaJaYF+/2HZFvxKkBIUGa5pNp5HPpwaL
z2EXJe1wEDHi7vJELpjPX3VQ+DKyru2ouU09F0FTfxitJFDmZmcyKLi9Lfj5+Ju+
aZ/4/5nkbZb+mBJiKZBfXcY6r4lyz+SubxX1dsCd/RQUYd1suKpjFU1OCpIXZ/9A
xiDOZJHcx47JWpNJpHvBV14/YCa1g6XCdgoCG4xUCrfGSHN3IgZUfPawDt6NGjTL
YDma3s10XBALNGLh0kRYt/fhPjFahFBGhgbLkygmOD3/2CWKaiotXC8EDkQS4uVb
Q8MHl+2+erEdeSekG3p3EPJNPGIxSKhR8UE+Omsx4XLR5kzt5gkG1reT2pX2YqQI
ERaPixRtjP8lqyiYTudUH6VKgV3ihDENEkhoZNge/G4bm7QFsEQbLueP1gU3MJkK
YuAV8ivIpgP6+vRPz9661HJbWYtwtgEQHOyrtXhYZf/+qymtiZMYhcMyMozEHaN+
yD2NL6zk3DS3thrCrGboM5+k9VLtTpGQWmQSPTpDslZUwXJiWyIoT7cwXCS6nchW
TDOn0LA8iC4ygGtZE4edKnFY0PUF32n1SGTNTds3hE4BJ/oA5Wyw9cDbVFStgYmH
sF4pQJ8AJ0KDeNymSvWvfGkPchVB9A8jrP9IBBXVJmtVkfM9ZfN6iGBNXLP8wIXk
Asb934jmHSgvKUT/LvFLUVe0AE64t1nuK+fSfXHxuC1JVCeQHssKpYT2f0oVFWJv
8ANCrzaEhqr+TUTIS+vWh8gmtzaxgjyPnvZ5995OaeJ1pxvOWqbPcPPH2Q1pkpM4
YTg0G6Ec/1kOcur2TyKnVyF817dQwy62/kXT50w/CTTKA/NR271MF6iKmwvmiTV+
6L7D4hE//YQyoMmU/Pig+jUGm/8kXnQ4CcOLf9iPa1g9Sjrj4Z/JuVcfQQ37k9yj
1jHO/4tK8TmHa1HihmAQr9I6L3aXhZXDBGjOSWl/9QDzxYAGdwH957ItvL6c9Ue0
prZjLcmRJVUvzF1BzrB83pEgbw2lkNHw/LW7253dIxltsj9DqLy0+GvKej/HJ48z

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
G63q3fhieLZ6uQ9mA/H4XCA6iy7pgrd/HNWamgjiKBly22D7cb7GnxEb7idBAZJq
XYtDREBZpCvCo75pekAd/ORPr4XQY2wvsymykTupzxTEbEeMx1Z0gg97VvSnj6my
ngXnb5UDq1DICxbVOAo2Ss6a0oRRtAzShg5Zf0DXOKE=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=11888)
`pragma protect data_block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`pragma protect end_protected

endmodule
