//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////



//This module takes AXIS packets on the input and provides a buffer to store the packets. These packets can be
//output from the buffer in a couple different ways. One way is just to output data as soon as it's in the buffer.
//The other mode is to wait until packets are completely in the buffer before sending out. This module assumes a CDC between input and output. 
//A DC FIFO is used at the input side to accomplish this. The output mode to choose depends on the interface the output is connected to. If the downstream
//interface can tolerate the output data valid to deassert during a packet transmission then the output mode can be set
//to output data immediately. If the downstream interface cannot tolerate the data valid deasserting during a transmission
//then the WAIT2SEND mode should be chosen. 

// Additionally, the input and output data widths can be different. A gearbox is used to align the input width to a central width.
// The centeral width is defined as the greater value between input and output data width. If the input data width is less than, then the
// input gear box does the alignment before going into the DC FIFO. This is to prevent any backpressure coming from the gearbox.
// When the input data width is greater than, the gearbox happens afterwards, and the DC FIFO aborbs the backpressure so the output data rate
// is constant. A AXIS reg is also added to help timing with the dc fifo, as well as fix the "cannot wait for ready" requirement the dc fifo logic 
// implements incorrectly.

// Two gearboxes are instantiated, but only one should be adding logic (the other will be din==dout and will just be directly connected wires)

module axis_buffer #(
                    parameter IN_DWIDTH  = 64,
                    parameter OUT_DWIDTH = 8,
                    parameter BUF_DEPTH = 256,
                    parameter WAIT2SEND = 0,
                    parameter LATENCY_CNT = 0,
                    parameter DUAL_CLOCK= 0,
                    parameter W_USER = 1,
                    parameter ALMOST_FULL_DEPTH = BUF_DEPTH * 7 / 8,
                    parameter ALMOST_EMPTY_DEPTH = BUF_DEPTH / 8,
                    parameter OUTPUT_SKID = 1,
                    parameter NO_BACKPRESSURE = 0,
                    parameter OUT_W_USER = W_USER,
                    localparam IN_W_USER = W_USER
)
(
  input   logic                               in_clk,
  input   logic                               in_rst,
  input   logic                               out_clk,
  input   logic                               out_rst,
                                              
  input   logic                               i_axis_rx_tvalid,
  input   logic   [IN_DWIDTH-1:0]             i_axis_rx_tdata,
  input   logic                               i_axis_rx_tlast,
  input   logic   [IN_W_USER-1:0]             i_axis_rx_tuser,
  input   logic   [(IN_DWIDTH/8)-1:0]         i_axis_rx_tkeep,
  output  logic                               o_axis_rx_tready,

  output  logic                               o_fifo_aempty,
  output  logic                               o_fifo_afull,
  output  logic                               o_fifo_empty,
  output  logic                               o_fifo_full,
  
  output  logic                               o_axis_tx_tvalid, 
  output  logic   [OUT_DWIDTH-1:0]            o_axis_tx_tdata, 
  output  logic                               o_axis_tx_tlast, 
  output  logic   [OUT_W_USER-1:0]            o_axis_tx_tuser, 
  output  logic   [(OUT_DWIDTH/8)-1:0]        o_axis_tx_tkeep, 
  input   logic                               i_axis_tx_tready
);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
GtRF6nNInQFdlz95CvwUgUtJbte/pcgNymK5Im35dwZSSBpWlNodxK19IjM87jfs
oOTWbMDiFKekU/50k5a9VQf2lotXx7c6pWfYCm5T2mpG4q/JwWotztC1P4YGInLP
CXMVG2CB7LghzqZjq64FZVLiBUbNBPRdsJXTMhTaojBhxXsKdSvA0PLfqSS07uLK
+V0wzJ2FoenQ33/rIhvLM+PSi9iZNSMRwj8D9A/Ju4ifv6ougdnurNT3mL0OauIw
/2r/0Q1RbRBcCdKi4ehl5G4FXExUw6GAc9y4VaG5937hhelGA8wSsyex3o+n97Tb
PgmzNuO0NqFvwW0Zztb36A==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
2yc3bIx7AVhme/Qp18CgTZih1DxFmHFd0RENEuQuR+QIDr/BMXpq3VFQc0bSqQIT
xfqHEw2fftxinSOsaxSr0D/jY36EdkDjy9HTdLOkC2i8YU7J9iZijLA9dDp+F+rS
nYZQ/xpwX5zVvMm0ajrMVs2VHs3lYOatMEOUa8dtZbQ=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
2g+G3IeOK8smJarnR/IffsjJCNh59X8HrQmnMJXVVakdZ8/FDnonOI2zIBJaAlzM
woNECTIP2Q6XrpW4OQOh8r796zakKfLzmbm9cuXMVfO6brlZdN1vi+pbfM+DURKX
5IIrh8kdcHoFyP+L/YsBHfkGOvxQa/qMypHyh/g2dZAwvHStVmApL6M3rL8ZnHv4
eOXmNfd64lCrh6m7E4ixyCkYAGkiR2ldTuLC/Z8k3IcbLTNqcigtFDAUM4AMDnIf
0Ddfq97DyxPeURwnS3O97fAMvoyT6nhrkxw9dTIi1f8pnM5KEBoxGF+IcdyfDnG1
6inTSNRquCIbp2KcakmJ2w==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
FBOch9ZFFtfNLsI7X+nBiAYZBFmYz1jqE80UmxjLd6vn3TeKopdraF0vpC1yGeDJ
93K041sYQEEj9iG3XbypQUeIZKRLz0iEE0dv4p/jUj9Wtny1V5Wgz1rKp/YytbIN
6QWTCtp51hii/Zi8PqpEtu9Ad29U/vAHQWMFdDyvVCOiB4ZNjnev0XlRXJ306jbN
HltXU+xPJqya98wFAfrEsALWsyeTuYLyG1xwM+w89lcrqMRSxF2yBMEgNKHN5oIa
/7Hg3pGK5WlOZnnEYdnUhY10cCdOTtjWvl/+gFmVpxkr9xQdHefDIRwPLPiI/zk1
TUWOst32lVHqXRLz8kgL9w==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
Ajxwrt1cqqSWTDf6+2mEbMEFq/T4MpY8qprfHbsn57hasmzxTXW/LLW5TYYOfIch
xp+ZE+Dnytn/yXGrEs6g6d+u/QTloDEbaAY3j6ZQ19W+n06Z2IRHetBkd583WVUw
OjmrmvbroSGBmdu+xEyCiXcc/IFog5+F/uKrzPENnucKprh1H5U6P65Oeiouitnt
5jtYs+ZiH6YcidAQkC61JMMXQhIViUO0+x6p9zJu/G9H9RMr7P/qhu48Wheu7pSq
NVv3qSfxYWv2XkL13PYY1GJLuH5qxdyZX5/qdoeYDLw8s8VZJXOrxnvkqeNcu3Qg
UKWVEcb7euM19YUG8nCfIA==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
qEA5DgQVUWYPxuku9yOPrqt6+XLeSfKx2QCW9/8IUytqST92J9NBk0XgBxTqj6Pt
Li4r1tFdwWrj2ExxAx7HY9nj07XV0ZLZvUMkDNhoBNxL98xQB0wKcyBEY5phvSEu
pueICfA8Sp7P2MwPvfGSXFSa8n2WuZSeNk1oikdADPBH96DgaQcCwOvMrJuCBSwa
Vdiohos6WAtA68dZHPlp9nXfLlh44DZgzzXUBcPNNf7zXT8o2tRNgD52T+lHXWLJ
XaBomeNsCe2cgg3DjGZm0dBkEpcl1xAJF0ZptwHL4qnwF54ejO4CjwUYt6f2iLfJ
qVyw+cILQNgtzL4Cq29zHQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
C/Xf2P5m+3mDnr2pFJdsRKjlemB0dXWe24mkxz9Op+iXp4wcLEPJUEnmRAbstkra
Ej7xZeOl/EmAXTt19NLuZtEs3dIA1s++XvzIBHwPYjU2Bdl3GyChy7cdaXNOnppi
kV/9VoEHZWQJY7pVNZuWFQQKZLaaRMeJrhooK2K1P5tSuyUKWMIngllqVJNZa24N
c5GDvZiX46wSkf8f/1hyIsZ0BNRAFME+xaXlx6H7AzqUOEavtSaLARehtKkBoaAP
CqHcFPkF2O+2EFOODrUvxdofLPHRlAYsl0qqOdqDl67CPtqVVMYSD6mJt7RzRnFt
DQF9lF+odlVVuuXKVYSU2snCTx8AwLmT2IKq3fiHD4Grmn2843kI+yMj2jWbqtWM
Uh+kP22WJ4Upm3pMIG6U7M4WARUBfMdFn+cmjrxqg+lCLPNjz05kHArFzxQ0v/N0
JKLLyD61BiAJzdSWhEMrWOk2Z61EOsRo5MKkcrmocUYNvci6O1Tc/TzEpE2yPUSx
B5HoBsGLHRDaUk/ffXR55gJG6p2fWOToTMoc8Qyuhu82Zx/W8T+txcvqXoZFbdcO
Iqx+C4qJygOy2ZUqugHgoXLtDI3xsndFX3vPvRseoJZGGe7nf26MteqsgiVzNN3p
drsmiFOJDy6noFdXJtkJY3DdEKWc1mSbmsaNceYgDAFoVTv2Y6M0smMo5CsBfOcm
QJISlXq2Z2sPJDagBoTKRCz65XVmNdNEUWBdXRQ+pykzQjR/L9z5QYcn3uTuJL58
yw06F99Od0gPrM4EcJW6fvdTkLOz/NLjxztgh0lqq8k1MOw44pKUwMimwaTYkZlW
2EO/nWRFLd1OW5OMcwIXm9g09wpcXxLBXKEXa3DddxlkUCnHEZUjBNReSZ+Wq+Ij
K45d32fxo3Oo0HgLFkMREIHptqZ7wACDmVELNbRYZJFA/9L//uPETqHhugadZrnO
Y6lGW2siGIOQrI3Od4jeLCx1cI/j9DOSuKfvoDW460bKxNoiW4AxMlqjVg5Ps843
Gc28oqSFpT9voEnAJ5To4bz6EZONxk/jqGGMp5okO2JyBd1AzykUgG16yroy0cr1
a681A2olCrbqMRP9AeRMb2Ib+FwL6tP842xD0QwhhAGa6Kn6/7fhmdwzSdP7pltg
H70ZVexlWPBSDoe2YIcESVeJ2F27il0hxXS0EJIWm7wbWuW0r7m8M/dfn0gWrPi3
1FtEZMRYCmiToh24AwsfCPRYSQLDJkclp4mCW0VErPXAVuGV2srFpj5XN1vpUFVK

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
accU4EYl6EUHdKikEaRNYJ+A3KHsSyTMentcB91p6t9QvK7WDA23Gw/DBX9kEJ+o
PLnIlcdtp6GytirsPIo7CPEht18GWJ8rBnG0oHbuIifBMhdlVp6ckxP7a72U07LF
6QPOK00ILV+8gtDS61VbC2BidgcRtAH0BaDzLufo1uQ=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=16784)
`pragma protect data_block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`pragma protect end_protected

endmodule
