//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////


//------------------------------------------------------------------------------
// I2S Serializer/Deserializer Module
//
// This module handles the actual I2S serial protocol for both transmit and
// receive operations. It supports multiple I2S formats and bit depths.
//
// Supported formats:
// - Standard I2S (MSB justified with 1 BCLK delay)
// - Left Justified (MSB justified with no delay)
// - Right Justified (LSB justified)
// - PCM/DSP Mode (MSB justified with no delay, single clock wide WS)
//
// Features:
// - Variable bit depths (8, 16, 24, 32 bits)
// - Stereo and mono support
// - Configurable data formats
// - Simultaneous TX and RX operation
//------------------------------------------------------------------------------
 
module i2s_serdes
  import i2s_pkg::*;
#(
  parameter MAX_BITS = 32               // Maximum supported bit depth
)(
  //----------------------------------------------------------------------------
  // Clock and Reset
  //----------------------------------------------------------------------------
  input                    i_ref_clk,   // Reference clock
  input                    i_bclk,      // I2S bit clock
  input                    i_lrclk,     // I2S left/right clock (word select)
  input                    i_rst,       // Reset (active high)
  input                    i_mclk_en,
  input                    i_bclk_en,
  input                    i_lrclk_en,
  input                    i_lrclk_negedge,
  input                    i_lrclk_posedge,
  
  //----------------------------------------------------------------------------
  // Configuration
  //----------------------------------------------------------------------------
  input  [1:0]             i_bit_depth,     // Bit depth selection
  input  [1:0]             i_data_format,   // Data format selection
  input                    i_channel_mode,  // Channel mode (0=stereo, 1=mono)
  input                    i_lrclk_polarity,// LRCLK polarity
  input                    i_tx_enable,     // TX enable
  input                    i_rx_enable,     // RX enable
  
  //----------------------------------------------------------------------------
  // TX Path (parallel to serial)
  //----------------------------------------------------------------------------
  input                    i_tx_valid,      // TX data valid
  input                    i_tx_ready,      // TX ready
  input  [MAX_BITS-1:0]    i_tx_data_l,     // Left channel TX data
  input  [MAX_BITS-1:0]    i_tx_data_r,     // Right channel TX data
  output                   o_tx_ready,      // TX ready for next data
  output    logic          o_sdata_tx,      // Serial data output
  
  //----------------------------------------------------------------------------
  // RX Path (serial to parallel)
  //----------------------------------------------------------------------------
  input                    i_sdata_rx,      // Serial data input
  output                   o_rx_valid,      // RX data valid
  output [MAX_BITS-1:0]    o_rx_data,       // RX data (stereo packed: {R,L})
  input                    i_rx_ready       // RX ready to accept data
);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
HwXvHOsNr+zNwXF4IOXQ0LCbPcQKZRflkppe5HHJndlL1dNGNIVQmkNh28uDWBV4
cQjxUxZOLgR2WX0Qgi7uKpmvwPpHREaUaHjXAV3rNbvk2GUqrxIdyNYrQbYb+j4U
GWuaY2C2jz1tV/7gnCajIsPULCRQEXozamz0N4nWbVUlqd3H658vN9NZ1CdFVIzI
bR7izuEkyGyQIU99xMeZFn3OXv9iouezcSuFD1nPjuQylgXWePwMAQa86HcO98Sd
fsMaE6K39AikylqHnRc4MeJYcbnU/Q47eMBGX5FaXyy9tPK8NJw+k2v/XQHNNyG9
0383z48uS6BaMP/XXYjkDQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
P4r9AAut0STun3ouXtvlmXaKfG0A4iefG+OKZ2xHvKCMeZZKVyMYgh1LUskDNCEr
KPZrqOExaM07r7wohNiA3SY+lq4ujmg+zwvlvEftXTHjQMw9aSadu8NdBY20gvjG
MKtk0ASipDoxXTvVFml4lBqLnU5hmwID5xZ6vCG6hwE=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
e9XiALBu//T9/UzR98t2RvGX5m323yqLsB+HSM1YkC6cW+p/fKnef6RiEQJ1wVep
5GCm6dk5po6KaQWK97s6DT1UpcfxhTedUNXMT9/xIXwkVRIZRZO9GCBUBzrICaQ1
yXzeenZVI82L95FRHyyuoKeMBUlu6SdsJEY5gKCK/lJNGknd1du14Tw77BU+X5xH
uEDjULR6ZfbQCJf2EDFcq9dGBKVinhnlppZbOedeFBE+YNGQlJ9lx4YLJXWgVbRA
t+XUatxWOJ9C9lNc+nxGHOs0oe8KBhOn4z/rwKNdljeC8AelTiWyFCs+KtvrLNUY
ale6T6OCiTfUquKqh/NgVg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
JUlrcmOi3zyKyqopQVMFbFakpBuViehvtavsjgzREUNzCgnYbo16prQh0F4t/yYS
+fAJSsOdAlkkhXAYnsl/pfEWVyMwzbdSav0SRYGqA27XrTZ+duHVJ3/wHPhVCM5H
4ICNVJYYVqSC5vxpY9zc0IomubIKjKncDCOgthfsyTen5NTYGmt5FVeEdIXVxckX
SDTVZhLsxSIl0xg/iMScegwflqOlAhujbWwfVwXwDqaasFSdEbSubBs4POR22x5O
TsPnf8LhNs8mJIce1HtXZuwuIiLGlX6YQmhrQWxm1VmOkxrAqZlYaGeRwRBcNz9O
bG4aJMdHN1MbeswY0goGeg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
p31pzlgVZRz7buvtoK5MdW0++YOxJwOLplWS/fDuIQ3f71fRSuildJ8QqaILeAYE
JiawZvXDN21vmGlkTn0AsdFd+J/L5hdmmPff7IadMWN68CUWHGf7kz9bip5JzYDG
u1T6gG8kZ6e69NqfcCR+eLaQk00GRW6j8/gs8N1J7y9Rh+OwGO/NiaZxqV7aMBut
iQoiwckLei55BwVW1Iirk7Jak78j8LyWpq82AJXa1nho8kLpfAW4gvXaKjRMyBsw
OZwzbHOp31pQFjahaAibhMVP+2i/ExktKSmyzkoAOvAAPprlsxVxA/JbfSgIT0vb
3M7fwhhkLhSIqtXQGADNtw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
IoQxB+oOD7eqc8aZBgBSycOp+6XsrLQ+2zDNtUuare14gRs91/wRnEP8lU1zP7p7
GrC6HXqsZn4Qmj3IUcj8C1NGvwmZdbu3g5+Qgi0bkrDftFRl5M6XnJ3Fpnh8WR3E
1u6ToECwUOdhBKrqKJbyVts2DJEUgRLrOZQbEh6/leTf2c3jKc7BwEsXDz8eix5O
D4EjD2EzOc05vAy8fTFYfnztUlU9yp8dmyiaAwzv+egnzVHHQhPRsH3Jd3lBW0/O
DIBecKVtJ1/3V4dk3TdPFynmntCbuJ7jZCA5N6eqWNtDW15KmcS2Ps1C7ECQY0RZ
qeWESci4odW7vOPik/ouLw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
b6P6p2+ZXnnCekCYGnOabDbqp0Y5btlgnSbxgS0JhbCVqCm7XpkkKPa7/n4jtkUE
78TQjm9+3b9EFZH5nOyzVJlTwdwuBSdnWx326pOCdSJ1C64L/KHOxk3gG5Xzc8n+
aOUUUB45nLRv1hIpJtJ0ZgdBZSeXzf4gUG5UFn64BRnDrPhvZsI8l5M0gUjJzANI
fVbYa/wl0AiQY0rfWNrVBVp3cqWYkl1K/rdJPvD8F6aZft0z5hxEnrv5X1wGHsKa
63hX6YZhTgsvpMq8d176VxIH8r0r66D1l/oKL3YDgAhwhuL5HU0nOdqRG5qwZYkv
0nwDMAzEjPAKe7Q42JSRP68mFZhvOQbvCl69OezKxvg+uuqODOrrG9j6ZG7fPyiZ
xuyMJmOUq86ytakRd1fGqeaROd8esWhkVb6H3dW4/iIGyMKOJSxGN2OZdgNYvbI2
PvDXSTcqIpmV2S5f3sPCBH3VW1F6XkU1MhotAyNkiSnyis8U5FL/Yy4aK9COHOWU
B1z0+ypfQtBJus0JCJMN2cKI6OUUfa96Lfq1XnRWtdkPOKfaES3ho7+NK/Ip/b2w
O2kypiqL6BsCAjI3DsfyLrxx8wZ/Vd/xX5b8BhjvjKvBv9Q9KRXrYgHfFEIXcBfG
HW0ymsKXFrYNrqyowzejkfUGuYY/sdZApCNh1vqavuycz7YDuVsaHFua5W/uDX+H
QJGTUUJyMBmHOwCl/P2uOaUf7ZnnU25+A8Fd8SjuvVf9LXOERlRoqlh/8X3QTLxL
1/Hbc45dvAKqKOVhM7BvY+0hQ+X7C46KMdXOYlrZK40Lgy1NECmHX78gYVlwyo7P
Zv6QHuYzf1WeGqy9HRTEqP0xqdjDAnTwrB208g395R1VwLGc11PNbNuqEIk1UkdC
3irezFwXYIXe2AXY3YlFhYtCX289yWJhkrW6VzSH5ed4W0ah2yT3uQu8KpMHjF/x
o5sGOrkPHNalp+5JuR8DeOnz9t3ttAcqwQPJoDp1n5MlYnmAruebPvRForqumL3A
GkUOxHBI85Imw6lp+cMeqxquayYMWpaXAxZI8OqlJf8LS4yBQ3fCusDQDygPhKnZ
ewDlg703Sla6wnhQebXyRAPfptfxroCYUYOWfwXMzw4urH+NZceblPDR8WUrfMWY
CZYsR6hLCZLv5CZvSLIDmxyZ4/Sg4zl5pIMnUZk9iYi/5QW+uAy0hIZI45nDXb7D
ln3elwH4ppq4gdvwlCcBQxMvswDmhlHufbUKb6KUOre2rGvyTPElPVF3RSWWLr9y

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
Z4+DEvIRLyqT4Xlq4e/CXTUd5YKmWRZG5Mr/AoRwzxuHlLuMXCwTJUjdbTs5Gle6
cpC1hpwrZyEOsdu7x1qtVSpx8d6fgsZgyw46IL5I5NCYwW+gPEXZqnOpJqeFB9pz
VWIabPorJOTxgysGiem+qMdNTjlY2rrdZNRkVxdZ9AU=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=15952)
`pragma protect data_block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`pragma protect end_protected

endmodule 
