//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////


//------------------------------------------------------------------------------
// I2S Clock Generation Module
//
// This module generates the required I2S clocks (MCLK, BCLK, LRCLK) from a
// reference clock based on the configured sample rate and bit depth.
// 
// Clock relationships:
// - MCLK = 256 * LRCLK (typical for I2S)
// - BCLK = 2 * bit_depth * LRCLK (bit clock)
// - LRCLK = sample_rate (left/right clock, word select)
//------------------------------------------------------------------------------

module i2s_clk_gen
  import i2s_pkg::*;
(
  //----------------------------------------------------------------------------
  // Clock and Reset
  //----------------------------------------------------------------------------
  input                    i_ref_clk,       // Reference clock input
  input                    i_rst,           // Reset (active high)
  
  //----------------------------------------------------------------------------
  // Configuration
  //----------------------------------------------------------------------------
  input  [31:0]            i_mclk_div,      // MCLK divider
  input  [31:0]            i_bclk_div,      // BCLK divider
  input  [1:0]             i_bit_depth_sel,    // Bit depth selection
  input                    i_clock_source_sel, // Clock source (0=internal, 1=external)
  input                    i_enable,            // Clock generation enable
  
  //----------------------------------------------------------------------------
  // External Clock Inputs 
  //----------------------------------------------------------------------------
  input                    i_ext_mclk,      // External mclk
  input                    i_ext_bclk,      // External bit clock
  
  //----------------------------------------------------------------------------
  // Generated Clock Outputs
  //----------------------------------------------------------------------------
  output                   o_mclk,          // mclk output
  output                   o_bclk,          // Bit clock output
  output                   o_lrclk,         // Left/Right clock output
  output                   o_locked,        // CLK locked indicator

  output                   o_mclk_en,       // MCLK enable
  output                   o_bclk_en,       // BCLK enable
  output                   o_bclk_posedge,  // BCLK positive edge
  output                   o_bclk_negedge,  // BCLK negative edge
  output                   o_lrclk_en,      // LRCLK enable
  output                   o_lrclk_negedge, // LRCLK negative edge
  output                   o_lrclk_posedge  // LRCLK positive edge
);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
wj+ELMcmSnMZtVSjnTLTzgXwM/I5m0V8PFvKgNTA5uy2WOw5u1SF8x4TiBv8qoYX
KbqQsnJhr1Q0WskSFDqwt+20xx7loNnpK79xivW1bmP20jy6WtiRkQm77nQMmpz3
qIhjPgWo/fyX86Um7RE6NHxM1gEo87iNdTKSjxrvUrS/naYgjpftnKMhJhY8JnfM
XjUXNA2Ey06GdAXgScQVsRQwWTOjF+1hJAZHSBQHmHPIIFAdrAvpSLhHfDSM1xOc
mrJuHmFV7Ry/oVMtxxu54vvjm2dmxlw5bYFrbTYqt3EzqJaxKn2pTiuBPeKByOog
imoFKE+PpnHBbpl5wZHlog==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
f15Os399CGfokbBbF6oOQ4UHw+4DndhjMmAx0i+q+olax/cFZsRNwuf0mgLgrnDN
3ZGcMfkqLKzgH7BOjHLoX3c+03nraostLX/J8d8H/fN/9GyBZySRBl08OrZxImG1
SQQfaEDkkwbsA8FPkQVI+UqmIDQ+xNjzs08zWiSPq14=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
TpL54z5oOT/RYuBVcc4ncn8NiH4DQVZa4sB5FJh44z3bPX8941ZktKp2Lx6dJL4X
v8Vl4Vm07TMpM6G3MV3kMokEkXszXXeXtCeTfH3kEsDQxmxRMkV6lU0VWORRRz9S
AMwbfRPO/1Vg4FWwPIVXhDNxlKgrh3tBnuCCLTe8wW0lrXfZqR+Ro3qB9oPcOEBI
uK7AqBCgt5xJq9Zmz7y2MQFOl2jAVoMMLVWcHtIVJTJvcHzq208QUCEd1OOPmirK
Ky6DlCJbQ3q3tH/rGPcHEa8NEd+A/U4Lc3uU+PLj+i0r0kUzQvutphI7nlCb8wBQ
Vo7rCrw8WCP021yBdNqufw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
VQzowPGL4dks9SAFSdkWwFBy9o/FI+v4B2hmneUSLB8t3MK3CkVxXrBqMT5DVMry
UGzMmiR5/Y23vQQeyvt/Rt7aK9Yy84D/fcngy6w3LZ0NoNR6lubcvs3iwVg5tRca
STfXJdChW7xlDfjUWEuiUV1CsbOpFfNg9Js7WDhqiXVsmXyNP27r4lm4RGnOHqa7
hMcPEAVtc/JhtP19tUPxfN2HdhzkbLoaOmnLDsqoWgvZnARJWxaIFOmk9RPImuzN
lh6Hns0NzVMXBlrisUpRyUwznJZPcYJf0dED/gcdldR3weIWowWOoga8TnfUQRMN
0SRw4VP64NcXhGh/pYar/g==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
JUCIBuIWLM25SYqMDX/m3bSEMkGyPQWhnRNkxLyuLSA/p3H7jUYnGxe7atZy0QUJ
848ssVB7MYzIccoux6xeT1OMT/xULsxCR+Cr8iePK+frwbebfTAeIQFtBG4XVg/W
DF9Xm0p616T8HhTLrg82RJiRK+UZ39ZfypyPVN8+R6K7dCAu+6KG/7uItG0rZDAE
/gsPHGf3TNYIUKcbqLlrh/cqievLy3EHxvVn4azVJxPVwRHRALDPari/+l1ufEvs
9xIC68tmUrKygzZO2FxjUkcHFLkLn2jutXZ85mukc/f+mrEvKN/zCAOrd9BdLrgf
IeyZl2KsHB/Y7ZHgYBAZXw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
uZY0xSPy+VWK5x5rj3cvXnUvifMsy9Bvk5fac6Hy2JHkQexii/gxKBm+0HAwS3v5
CCgWAleYCbfnZqx6OgXIWT21jwi62TGKZ+r7to1KrsWSpWNseugmMjFvR0YWFOx9
AJbSqqjm69uM1xfuIz9bJeOoIC3LbsT2JUKCJ8tMpk99CUzQ7EOXwGQE6xubIQ2A
G1hHhXfN994n9wbyJDGsR4N6k8IBPh8QcHU2+im0oXok8v8BgHDdArTFJA7GZkbU
uiPvjCB+G8l/E5Ea/vw5P4rPJxJTY6mSAHYDPycTXkSF/K9maB/Zx2ryrdmoV3tY
/XntiKMrPm1jdT01iFVOTg==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
SkyD2xqIfRJ+g3Q3xwgml9M+brboaMU2nJS/BwjmdJVQpAKhXJT+SOiSuCkVknCo
X9/R02k9JoU+xtg94ZKB8hjSU/D2waluc/1biOs1HixPzumnpQ9w5vGo001/44qI
ip1KXfvBLOBE1E9AZlV3ShT3loUyvw2VEl9o7dc52sTie+8oCHlgbGpV0923aaA3
a6C9rUKjJXywNBp+mlBrz9i97OmAwW3+ECgm8IRMrndep/hqBaR9s+nm4NTUGyud
TlQv7YJOpGUmfdC8C2Uc3BIhuDi4KwqRgSpSQvkWY9+UHSWCU58L2EsPfOLyHAbg
5pWYFofRVGZWXIfUl9GEOI3Wcr94ppTKOwu+uxdyYmkX3C7xHTH2QnuCFDd+yR0u
7qrl1xQNl0rHCS/NrTUMETyAS/7A+LsUuuo+S1TGdekG0kgmqv/ioW1+rFfQO2j+
4H3vY8fmqkvlEP4uS8f7MVCbPIVlcvnUf0xMZbZJFFJdWXykfT92es7s55GJ+WHL
nKAx6NCPmpP5rsY/mTt2z0KpKar6JC1iYkg25rkXK1Bv3BK8ZACjxfyUzfmFdfjM
ZRBxooHzUM24vw2JfIv0mS5V78kySQ5SEsjO3yUGUOi8JR8Myeq8S+t3dKBxJU47
oQLdbxDRr+jqhE2PY0K84MJQQA4nNf597rxgxDnlKZomrah8QYNAduiSNJ/D2K0i
ET0xdpLiXRqkafU2QeFICZ6UOtfID6C1U73JU2yddfQG2OwEvfoCkEB8veoYxSoX
aKMZ1qE6Y5RhHcUz8PuV+a0R67dVRNJym7hGjIn4ypEIEHxPADgfwoo0wh2i99Fq
zoqoXfUTRkiIQQBbSPU1R3+z34IuUR/u0WnoSgOPf6Qra3s7PQ7pZ5RVcuDZ/G8m
MxaNxfbOaPiNqiP44o3AVsydOPIDbqf1geoV8gleqxyeM9VTR0sYKjV8Qs2gLhmY
zv+mJC9oRILhGbKjwEYSHrKJN1YZIy84Jt9qc04RNLOOwG3MxcMRtEraer8QVYC6
B+l6+npt5thcWOQmcRPCkkn0RSRbNRqIzQ5Kni3yzhwmb5RWmEctNFSPCSLQIIpk
Pp0wEOtzcAEXrF5VvRtQ0Pg3UdVS7Sd6JzvDYetUjBFrpPJGDXStxmccpHt0ZDfl
kWsP2hE4NSo2yP3F2ogaglWQMPOe1AXONYDFMNFsnHfupx5yJUj7SPNRekM61hGo
717n15pYv21MQr5jYgnD9O81qc7xb0vMtqN/vPqyqQX9k9tGvMnx9DUYtSlZ+ZO9

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
l155nCTfjrfUP7uRjoQP590e9iM3Et51RQhJ6aRfciKf/WDAwWV2fIkNFIdwtwTx
zsqJSKKq7sWMuMhymhA+cds4U6dCyH6rHczfsrGZ7VlkCpYRaw/1uThfqJRa/M9+
9fqhEzAMcjU2kkLwzLEhiSIUtRL9+sRtYPD5vfEO3Kk=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=8640)
`pragma protect data_block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`pragma protect end_protected

endmodule 
