//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////


//------------------------------------------------------------------------------
// I2S TX AXI-Stream Buffer Module
//
// This module provides clock domain crossing and buffering for I2S transmit
// data path. It converts from AXI-Stream interface in the host clock domain
// to simple valid/ready interface in the I2S clock domain.
//
// Features:
// - Dual-clock FIFO for clock domain crossing
// - Configurable FIFO depth
// - Almost-full/empty thresholds for flow control
// - Underrun detection and handling
//------------------------------------------------------------------------------

module i2s_axis_buffer_tx #(
  parameter AXI_DWIDTH  = 64,              // AXI-Stream data width
  parameter I2S_DWIDTH  = 32,              // I2S data width
  parameter FIFO_DEPTH  = 1024,            // FIFO depth (must be power of 2)
  parameter AFULL_THRESH = FIFO_DEPTH * 3/4, // Almost full threshold
  parameter AEMPTY_THRESH = FIFO_DEPTH / 4,  // Almost empty threshold
  localparam AXI_KWIDTH = AXI_DWIDTH/8,    // AXI-Stream keep width
  localparam ADDR_WIDTH = $clog2(FIFO_DEPTH) // FIFO address width
)(
  //----------------------------------------------------------------------------
  // AXI-Stream Clock Domain (Host side)
  //----------------------------------------------------------------------------
  input                    i_axi_clk,      // AXI-Stream clock
  input                    i_axi_rst,      // AXI-Stream reset (active high)
  
  // Configuration
  input                    i_stereo_mode,  // 1=stereo, 0=mono
  
  // AXI-Stream Input
  input                    i_axis_tvalid,  // AXI data valid
  input  [AXI_DWIDTH-1:0]  i_axis_tdata,   // AXI data
  input  [AXI_KWIDTH-1:0]  i_axis_tkeep,   // AXI data keep
  input                    i_axis_tlast,   // AXI packet boundary
  output                   o_axis_tready,  // AXI ready
  
  //----------------------------------------------------------------------------
  // I2S Clock Domain (I2S side)
  //----------------------------------------------------------------------------
  input                    i_ref_clk,      // Reference clock
  input                    i_i2s_rst,      // I2S reset (active high)
  input                    i_bclk_posedge, // BCLK positive edge
  input                    i_bclk_negedge, // BCLK negative edge
  
  // I2S Output Interface (doubled width for stereo L+R samples)
  output                   o_i2s_valid,    // I2S data valid
  output [I2S_DWIDTH-1:0]   o_i2s_sample_l, // I2S data left
  output [I2S_DWIDTH-1:0]   o_i2s_sample_r, // I2S data right
  input                    i_i2s_ready,    // I2S ready
  
  //----------------------------------------------------------------------------
  // Status and Control
  //----------------------------------------------------------------------------
  output                   o_fifo_full,    // FIFO full flag
  output                   o_fifo_empty,   // FIFO empty flag
  output                   o_fifo_afull,   // FIFO almost full
  output                   o_fifo_aempty,  // FIFO almost empty
  output [ADDR_WIDTH:0]    o_fifo_count,   // FIFO fill count
  output                   o_underrun      // Underrun error flag
);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
zDBZ0OkhHFl7Hl2gnWh44vEFJdoHj5DNkGGm6R5ViCTR9sbVVjRBcpJ3nV+t3XQD
sgEcUWT8KHvcPLUU9VLweHdG52Ovy8ghgURXpASMintNLlPszzja+qrt3Szy+/5U
3T4eRqzAMzkjlCLkrSUhyxeRL7N+AC3AT0NbUMhIX8KhQ04yRNejP8g7gXulTDHv
nVoyRVLkQ9sDqleRKhYK6+z3uQrHHVha8Ty3qQEn2Q01fiy0sZpwNFGnlgYAUFPT
/yJc/1SEQeCnUFcBsOmJZ2yvE3Ql3yB5cYEVpuhytZwcfH6rj8YpX4SnojoVJbjg
goAThDrdtrpIraNHioNB4g==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
IeRyEA+vzTrcGQAKV59Upzjt9XcZnJJdSBWKIaGQFjNJPnEa4iqbiGB5y5JQD/0r
ZKl/0eOEA+WPXP4wAgs+BUo8F6u+HVeB1sdVVqH+ym+Y5dwTGm+W1tpLtLBERE2F
/pJsml0+MyvCHBuhAhvS4QciwDvwALAVoP6cr/8VyHo=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
y5FjmYYi9wlPEGIO22DVIujs6Ye/0NxfeTSJry+IC37FtqyVxp32zvQRQ4VY0ry2
aEZ/sQ0347y4mxkT7vyLmWbSDLkE9IVUG0H2bJPVYemfFKNLzDlFjhfr6ihXIe1M
dFIGCCsJxeUfu6+udgz8dJWhxHredTaVBVbLngxAv9W+LkDgpgyfBNp4RTC3Mp12
0rgwD55ATQm2pJz4Pf0HynBIPxE7Eayb4VJ4kaxq43t64xZxDFId4lVJF9ytrVIP
FJxCLzcTRFVTGNyv6GON+vjFvWQvw3TYYoTewVqjnpMicblomfzwxWGoUV1gvidn
wdswAf/t+u3n3RXiIc6swQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
BS5n3uiYR+Qg4YIc0/mA+Og81zP0vnl8OO02ckl+jeNJIfrxTYIvjwZETAUnCZyn
ZdK8+kbDMy6ev5Sfxpu570OejPvy32YxgtBMaG1c6K2R6zQryWJzGZ921/rgmM+6
+Ah4pFKcpGNadnuN+avitUX0KrlyYYHTnSjzO3GUj/MGu4eJWolzqi0gW07YQ2tT
B4GkvxKcdOoQJXMvg55P/fqNDHu26R8U91Mt14cF39MScwk6eZK9zOXFMpWnRVOd
6ShQFtLL1uSr+x0DBC/jIoumPshxrtQRaBxFmaJvAHuhHwQeQky+uxLH2KA6vcZy
9Wm3OHMRjc8lpHqn1AZBzQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
WLRU5jhtxN6+xR9vxgNSstfMiM1bRCpLzXg6ny6T9iuEef5wHqpVv/2huB5b0Ae2
7QYBLUeWYXj61GMIbaABDoZSRJfb95ifN2vVEFuzozK+8x9X8HD63j1Mg09rk8iB
6ji1G2WcdxV1ZiKZynzFWO3S85Ukk3M4rb5cjZWt/6459B+HsGq7idS6WLLG76AU
EuGSxSUgibI7rSFTTWCG8em5zYVfV7Kj44vxDp4hPdbejSOzJXBc93AwMX9AnfaR
HmhhVJuUSKPpYEgoyJ7Jn7dlT+mT1JTnlftmqPA0ry1/U+iMHgphXg5YJSXMyaUj
HJMQ/ieSpZAfCfB5gMcm1w==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
sfUoscmu6EpNrBSbllrpY6ElGznRx607LWK5rE9LqIaAwV41kd19VhH5RpxGL3sv
4WiO0v6QXp8rEJ8UHazlHQD+PIuBf2S79M0qW6PRuBXIBJCHxIUPWk/xEXf+GTgP
LhiO5/h49xWjMzfetHA8zEiquRiW8VC4a3ARqxSEQ2AyuRTABXmQmRQojbDsgfDl
YbVyWooIj3eiyxVwpTNpTH2AO1n0qDsesp8D4x+0wx+JslQPyp96g9QHgctVugWh
oi0gFAifggKM5YUGgATC7DolpXD6x5G/dxfBEsWY3wmGHcTw5dwfptk2B+PehuV2
3mpvwXIgvsJ18bVVE5d7Zw==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
cNvSNJoPDqYEkQlO/xSR/Ze209WS84huuScsdGC9CKq7G4GhXFX9IPhEzF/t9ARr
Ia1++JbMIMU5+ZuEC2W8vIgfi1nmvKvuXCpiIg+2FstlOI+zojo5lx2U/gTdpCG3
iBD8lO53WGb2cq+Jw8OxinVka0/tTe+JH3ZyBhGJIBlCX1c/W6QLjH7eNXe17XqJ
llAFOg5U1rSQoRHAbeMQnNhDJaj77gYFTCDMfwWyTS8DznZXIHEaPB7dOW75mwzB
OUAgNR5IUfaVtA9icox9kiqcEAPos+P33XIuxAXBzTZq/0Kl/h//sgglYryrWf7B
nXSRw+JuBnFVSjEQuM5c1MgDnAaQaLiG4ix0VAlpDHWVs6IAlUCXKvf9v8M+9g0P
57AhLhdmQXPiCd7+DTtyqB7miUGlfQ12KibzIkC5cmIMYM5z5yPPrRaWAddIWBBw
UYauvxjmCfW9Np2x5T98DFS9GirW7gTD0DNRFObp1jri/pIAB12pTkJuQUfbvik4
f7uXSsxSkEKbs/aTbGgH561Arxxy8LkB/HljTilt5raYyZbCTHuiuGY5ecwenF1/
CkvQEB2O4n3g+5JnAJXH10Zg55Sw83LvDwIFXqIB2QLCiWTZzN3vHEpl9q8ZUzhW
fS6nqNeA6LWFIt7W4CH8YjcvUszW9M3cS5t6riSv22G49TU3xpBV6ZeyLBON0QhW
7d7TQmwDUPTkIw/lrHeKbpcNMtGWwmkED6d8IuJtc67MvE0/fFPazMMIgDZ3HKnE
68U8d/QWgQK3b4wBRSBdKkCxF+WHpIJiAHPb7aaPIjFeUQJ7zY9WaovXqy6aF3y1
faZafNCg8hLajMcEUS5LcboO7ZQufl7x7moXXmoH0OdbqOiTWwqk5KuVoUMWEQjk
6PRoQ+mUDWiNoLJiUshzzi2CqOOEOlZm9HzBC7HwIgHCaMrFUOOp6CZJphzP/ck2
dkthy9jgtSRBuM9+Hp2NhMtiVmGF8/dV5eEpVMoTPv6EkMlL7wTFZhXamNE8FTgf
W4dykOWv1QyHuVmCppq+JgnqrEEyJAWMHqbYCTqtbZ/eqzZt7/Qlt6AV3rKGHIim
8WuEraE56vmsZmyYI/VlbJ6kN3McSuh7r/nRiBlDa+auH9CQG9/eKsgELcwfqoKR
MJhPU9kKfU5lJjjLN81u0aejoNV/qlbxv8uAYn5Xmdsi6PBLSRv6P0X4Bjp7JqQs
BE7FMlYiKIH8iJaYcPZAt4AT/hTaUnPpZ0TOiUpir0s3sYV4oXxUC6uKqOU/Hsun

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
JZht9pWreOSo+TxQisszZm3/PPO1ag4WJZmG0MLGXAaZbX7W9pM9S5SdlZwfOupG
jXDpe1Q9lRqpl0BBabkIvzDY3HAPTXN8/DhV6/0rJ/jPfvFVgMa1vYXiDOMuhxIl
wboLxcJneJh8bdiz+HfjHiAjD3deu4VGTz5bWsGn5e4=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=8672)
`pragma protect data_block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`pragma protect end_protected

endmodule 
