//////////////////////////////////////////////////////////////////////////////////////////////////////////
//// SPDX-FileCopyrightText: Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//////////////////////////////////////////////////////////////////////////////////////////////////////////


//------------------------------------------------------------------------------
// I2S RX AXI-Stream Buffer Module
//
// This module provides clock domain crossing and buffering for I2S receive
// data path. It converts from simple valid/ready interface in the I2S clock 
// domain to AXI-Stream interface in the host clock domain.
//
// Features:
// - Dual-clock FIFO for clock domain crossing
// - Configurable FIFO depth
// - Almost-full/empty thresholds for flow control
// - Overrun detection and handling
// - Packet formation for AXI-Stream output
//------------------------------------------------------------------------------

module i2s_axis_buffer_rx #(
  parameter I2S_DWIDTH  = 32,              // I2S data width
  parameter AXI_DWIDTH  = 64,              // AXI-Stream data width
  parameter FIFO_DEPTH  = 1024,            // FIFO depth (must be power of 2)
  parameter AFULL_THRESH = FIFO_DEPTH * 3/4, // Almost full threshold
  parameter AEMPTY_THRESH = FIFO_DEPTH / 4,  // Almost empty threshold
  parameter PKT_SIZE    = 64,              // Samples per AXI-Stream packet
  localparam AXI_KWIDTH = AXI_DWIDTH/8,    // AXI-Stream keep width
  localparam ADDR_WIDTH = $clog2(FIFO_DEPTH) // FIFO address width
)(
  //----------------------------------------------------------------------------
  // I2S Clock Domain (I2S side)
  //----------------------------------------------------------------------------
  input                    i_ref_clk,      // Reference clock
  input                    i_i2s_rst,      // I2S reset (active high)
  input                    i_bclk_posedge, // BCLK positive edge
  input                    i_bclk_negedge, // BCLK negative edge
  
  // I2S Input Interface
  input                    i_i2s_valid,    // I2S data valid
  input  [I2S_DWIDTH-1:0]  i_i2s_data,     // I2S data
  output                   o_i2s_ready,    // I2S ready
  
  //----------------------------------------------------------------------------
  // AXI-Stream Clock Domain (Host side)
  //----------------------------------------------------------------------------
  input                    i_axi_clk,      // AXI-Stream clock
  input                    i_axi_rst,      // AXI-Stream reset (active high)
  
  // AXI-Stream Output
  output                   o_axis_tvalid,  // AXI data valid
  output [AXI_DWIDTH-1:0]  o_axis_tdata,   // AXI data
  output [AXI_KWIDTH-1:0]  o_axis_tkeep,   // AXI data keep
  output                   o_axis_tlast,   // AXI packet boundary
  input                    i_axis_tready,  // AXI ready
  
  //----------------------------------------------------------------------------
  // Status and Control
  //----------------------------------------------------------------------------
  output                   o_fifo_full,    // FIFO full flag
  output                   o_fifo_empty,   // FIFO empty flag
  output                   o_fifo_afull,   // FIFO almost full
  output                   o_fifo_aempty,  // FIFO almost empty
  output [ADDR_WIDTH:0]    o_fifo_count,   // FIFO fill count
  output                   o_overrun       // Overrun error flag
);





//put a blank line above


`pragma protect begin_protected
`pragma protect version=1
`pragma protect author="NVIDIA"
`pragma protect author_info="HOLOLINK"
`pragma protect encrypt_agent="Radiant encrypt_hdl"
`pragma protect encrypt_agent_info="Radiant encrypt_hdl Version 1.0"

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Lattice Semiconductor"
`pragma protect key_keyname="LSCC_RADIANT_2"
`pragma protect key_method="rsa"
`pragma protect key_block
cHKjEFJiyxvWfgs7YEjQGeFUrcsHMywuJZvgy+hISVs+bC/JJ0v07nXZ1sCISwMW
a9RliUMmM5FFXUVfRVM8fET4Ug/e9eSrpwyIuOjHvmHEdvJbdAhhLy11H2R5/Vew
RcwPFWUlpXDNTHAjVED8M8UuBbYWPdnuseLmm/+v14XkznT+jKlanOLaNSS3jZIt
vHh6RwF3iM0NK9il9YBGr1sbPSjgMhg6q3z8hZfYgSZ94cL9Al0y8NI8tAkobwzM
6UK7e5ZTJvOWi0+njylB/Ipad9SDYcAnB/Qy1gZuMz/7XsKYc+XPWV4bLhkaF9aT
XqpeYPH1LYrZzKbrHFE6ew==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Synopsys"
`pragma protect key_keyname="SNPS-VCS-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
ejR2mAMeZIRu+tc3Thdf+mriRZSB02OJ2KKKCTBRsdj1oU0SHZKAZ4FnmgitOwdt
YjgOx8/tvSkXBLo10PesPQnlaQi7Ku6zo2iNq8j3FNRrJiwqyRBqi5Lyf+//EqE/
ufHtp/ZawUuVCOJcoq6nXjMpSDgzICkcuc89lyuPogs=

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Intel Corporation"
`pragma protect key_keyname="Intel-FPGA-Quartus-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
qZon1Lzy9UJO0u27WVS/ckPCMpI3csnLvc/azfx7j0Igo9lUoh7Iv4Igux/qvLzQ
NgNf/Dv3xiCtMLtMo+TlSMzG4pLnEZ1cXME+jqLQxYdjZUZ1iMgO2/gpDLWZ8BNE
/FPXgkbGPRLJt7yZQ3JGThTnRo9TTGqitvRaj6pPLpB6483w1YZ7p3LPbAWthg+V
tfqOqIlg/6U4N61L4H+fC9Gpj3LoZNe8aIJj7KeWeeFpj/g68lY4pYtRxbp9kdnk
BvvYayIRcpRyXF+2mQO8msU1UH0yqb0ASUb9WdqTSeOw9ARZz9X9FbyAffUtlZTU
AnHnOLsE+zrBjuZa+wGFXQ==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Xilinx"
`pragma protect key_keyname="xilinxt_2019_02"
`pragma protect key_method="rsa"
`pragma protect key_block
U1xIiqFQkZGT8yv+2sIwMMjoE4sG4RiKzRE8DOQhaNa6iF9XtP2oer++2cE8YSdf
9tNRc7x76r80GzT4HU9Iy3SRQUxlZlrw5uoP83xfOlrFVoS6pZ//m/+dWx7PwBcb
K11Y5pYGIn46L/f0kGMKvX7enbPu0qSRlf9+B4zxUfTr7i90DbDFcspvQqN4CozA
e/WwMk7IS4md6CjBhcpufIR/D2xpMyzgEkXq1g2gtBqxnbOXl4R7syS7ovQZYtVT
pnj8Y/OcRRyeSWxRnlutopRMZR40F2Ka+dHy46f51t/lXSXM57yvoIz2qPhc0zjG
yJe34Eo6eS3UtSe1F5bp8g==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP15_1"
`pragma protect key_method="rsa"
`pragma protect key_block
ckxWNoy+WrvIfci4wSfQVdEPu6KxQNB4e3+tQmmH1J/EI9KuatjrEXNSkyW/0XvW
F2EFNfHlOrHPiCmbzT5lsFiTUIBdcP4tMxv1TO0axhEmccRpCGmdvJX6bBP7oLSs
tb2+aVj1nDkPLv3RkUTzs0E8lLeA/Q8HUMJ2KjNEFZl3JV+zyV/CaJN7sPTEmo0M
eaytMxDh2LEOEsNDDBih88T7omegZ7OBYnR1ULvC+33frmgnZHTW77P0no56KR/n
nlIKeCmygH39lzTLjyHCaHfT4c3oybFgaIZWhWNXbGjwajyDh+EL7bw3n0QBuLNj
q1VNm1jlCNurJmotNjcxoA==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=256)
`pragma protect key_keyowner="Synplicity"
`pragma protect key_keyname="SYNP05_001"
`pragma protect key_method="rsa"
`pragma protect key_block
xTfMdk3xROaELa4JDvOcMPTbCxrNF/SSK4fgGhGHE3570WAM5ybkrFcAZM96Z6gD
RW9haEpqVJLQETMCTMLJn7tCSd2bGXiPNBTY1uwqqkCcaeMVYRO2AxgXsOylOo7u
SkrBs8Eio75JrO1WW1Fs1nNOPtdKJq04PM+p3mjkoUOvh3FjkaIB2IOSHcWjSxF8
15pj3mcSoxcivh0GnBxOwhmdpMu6L/XoQJxuvzXlHjgC8m+pOJtI9hfg2qTwg5+e
AtljpNC3DyZ4H3vBJR7tGPV0DnacBN35cYTur+I0RUNFIpQTukgR/drUwLorPL6Y
W8woNBjaCvYWM5LLCc1plA==

`pragma protect encoding=(enctype="base64", line_length=64, bytes=960)
`pragma protect key_keyowner="Microsemi Corporation"
`pragma protect key_keyname="MSC-IP-KEY-RSA"
`pragma protect key_method="rsa"
`pragma protect key_block
xjJ9zQnQSgOrWKed1SXuUDQnx0MTR13OOA3wX3UmZEmwhlcimHEkuj9HMP8W8iHz
chHwbJjkXVhCN30lLGlZrg1adzVcUhLTZCYXZiGvSfuUQ8c3S70UMoBB0m4CR5pF
TvgicJafLWa7WsqBYzN2g3cfvh1kUplVo5pSWz8TIDE3VmLkfzQABILeLkEixFih
WxkcS2CevPWyYTx4wSF5Pp13tiWa4aK5gxh4UExwyQ3N/Zdp+yLZmPOc8vhA8vUl
aEOeUuLIl6vTQUdLox+srXQwjAMmaQp7aY9BoQ02cGe210EfVEtJ4RUNx46UT19O
wC683++wEf2toDKPmZP576JxEmOzum20u6wATWsom9Um7eC/WyfNXUeDxRysAXpC
lEbB7hUJQRZdDCLnl98jpibYF2AJLk1FpRYR61t3Keo9y+tCkadCx2IAQkDYPGgl
Shn2U4hNO75DtwjARUbnSWf8X1X5ikyqp9JR/KtqGfwla6+MwXg98c/GnHgKJYH9
9CO7JAg5YCMCFA7iqJbEreDXmxgMjkB/IYfiRlf424a8X4Kq4IUR2A9liV53c90I
owcNTZG13McqisIH4Ks7++zymK0MiDfVqeC6cFLUoEC7G3S3TLZFW91eJv+/QReM
bE+Kexkj9e68mcmkMTvMBXDi+JzRp5AFo87mh68giqYYW8aeLl67gKP9reP2aJBg
oiw8slFa/aa3idtXA1Oyk6Sv8zmmIVig/j/n5nevbPmAG/YwueeMu5aM92g11z5A
lkQ8SsAsL8SrJA858rA5LHE60GDxrljl/PB05OCXUSthgwIrak4/ZOGq5ezTq4xa
sWeux6TGANBR/rbyrBZJuLBfmB+49Zi8yX+4JJeXmq0UkOu0GCGSYWXB3fPkBfAo
gmRrV0Iksa1Pb2NJnpi4t19c4Sn2QzxBDQEx498sgFK99yfBD+kFcNhA1lo7kIpj
AvUPJ24sr99UJzm3tUZMJZgkvwYjWZzes7Aoe43ANwSUdUrGZtQxDXnmiGSDEV+A
IJbMjX33TI5nRGcdaoqRkWUTIcVuCJp8Qmwk8SjqaWunIfQkRPwKo8mlO/cjITbJ
cOyFxuKu4928mv25OIJV6cNyclKMOuJXcmrJdKWd33Huqos3K1gKLbQocx10DEr9
kbFcN41KLFqAGdSuf9zXyJIc4mu37lAl2Vmp/AfEyp3d2hTSMPtMDCqQ0t6j3A3E
/rJopGBRRX4Y+wc4Zlhe+FKrCWecBbTLYaibw1WLEdbx/T4dVwDALjlxQELkw7Lu

`pragma protect encoding=(enctype="base64", line_length=64, bytes=128)
`pragma protect key_keyowner="Mentor Graphics Corporation"
`pragma protect key_keyname="MGC-VERIFSIM-RSA-1"
`pragma protect key_method="rsa"
`pragma protect key_block
DYT3dHTvt6n5u7goFcXCtdF5p3Ko/v+LbNzEDjzBreXxK8c5xPPJ6gjuKaWgH7A8
gWt4P7AtNRFLtjh0RrnQb+J7cUlDQXt1fGYK8a/I4ne+BNraAZQpW33tL9v2feCx
ZtXoYPJ/v0gLAl8RKDoWfA5pw8AnS9AUCTEUcX2qk1M=

`pragma protect data_method="aes128-cbc"
`pragma protect encoding=(enctype="base64", line_length=64, bytes=7840)
`pragma protect data_block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`pragma protect end_protected

endmodule 
